; file: makeflowcubefiles.pro = vx, vy, divergence/convergence, vorticity
; init: Jan 25 2013  Rob Rutten  LA Deil
; last: May  3 2013  Rob Rutten  LA Deil
; note: Uses assoc to read input and write output cubes as files.
;       After Bob van Veelen using old Roberto Molowny Horas programs
;       via Pit Suetterlin.  These (odd.pro, flowvect.pro,
;       divergence.pro, vorticity.pro) sit in dotlib.
  
function limit,val,min,max
return,max([min,min([val,max])])
end

; ============= start of main part; adapt the below ==================

; ##### specify input parameters
winlength=2    ; temporal evaluation window in time steps (minimum 2)
fwhm=2         ; spatial smooth area in pixels
itstart=0      ; set time step of desired segment start
itend=100        ; set desired segment end; if 0 then to end of cube

; =========== telescope-dependent parameters; comment other out =======

; ============ DOT =================
;; ; ##### specify output path and filenames
;; indir='/home/rutten/data/DOT/2005-07-13-spot/Cubes/'
;; infile='AR0789-gb-0000-20050713-cube-1074x814x332-dt30-lo7'

;; ; ##### specify output path and filenames; adapt nt if only segment
;; outdir='/home/rutten/data/DOT/2005-07-13-spot/rrcubes/'
;; outdir='./'
;; floxfile='AR0789-gb-flox-20050713-cube-1074x814x4-dt30-lo7'
;; floyfile='AR0789-gb-floy-20050713-cube-1074x814x4-dt30-lo7'
;; divfile='AR0789-gb-dive-20050713-cube-1074x814x4-dt30-lo7'
;; vorfile='AR0789-gb-vort-20050713-cube-1074x814x4-dt30-lo7'

;; ; get cube geometry and file datatype (telescope-dependent)
;; getdotcubegeom,indir+infile,nx,ny,nt,datatype
;; if (itend eq 0) then itend=nt-1
;; headersize=0   
;; header=0  
;; bigendian=0

; ============= SST ================
;; ; ##### specify input path and filename
;; indir='/home/rutten/data/SST/2011-05-07-EB2/cubes/'
;; infile='fe6302.2011.05.07.cont.icube'

;; ; ##### specify output path and filenames
;; outdir='./'
;; outdir=indir
;; floxfile='fe6302-20110507-flox.icube'
;; floyfile='fe6302-20110507-floy.icube'
;; divfile='fe6302-20110507-dive.icube'
;; vorfile='fe6302-20110507-vort.icube'

;; ; get cube geometry and file datatype (telescope-dependent)
;; lp_readheader,indir+infile,header=header,datatype=datatype, $
;;   dims=dims,nx=nx,ny=ny,nt=nt,endian=endian_file
;; if (itend eq 0) then itend=nt-1
;; headersize=512
;; if (datatype eq 4) then datatype='float' else $
;; if (datatype eq 2) then datatype='integer' else $
;; if (datatype eq 1) then datatype='byte'
;; header=make_lp_header(intarr(nx,ny,itend-itstart+1))
;; bigendian=0

; ============= fits file (SDO) ================
; ##### specify input path and filename
indir='/home/rutten/data/SDO/2011-08-18-bello/cubes/'
infile='a1700alignint.fits'

; ##### specify output path and filenames
outdir='./'
outdir=indir
floxfile='a1700-20110818-flox.fits'
floyfile='a1700-20110818-floy.fits'
divfile='a1700-20110818-dive.fits'
vorfile='a1700-20110818-vort.fits'

; get cube geometry and file datatype (telescope-dependent)
inheader=headfits(indir+infile)
nx=fxpar(inheader,'naxis1') 
ny=fxpar(inheader,'naxis2') 
nt=fxpar(inheader,'naxis3') 
if (itend eq 0) then itend=nt-1
bitpix=fxpar(inheader,'bitpix')
if (bitpix eq 8) then datatype='byte'
if (bitpix eq 16) then datatype='integer'
if (bitpix eq -32) then datatype='float'
mkhdr,header,abs(bitpix)/8,[nx,ny,itend-itstart+1]
sizeheader=size(header)  ; fits header = Nx36 "card images" = Nx2880 bytes
headersize=(1+fix(sizeheader[1]/36.))*2880
bigendian=1


; ========== rest should be telescope/parameter-free ==============


; open input file for assoc
get_lun, unit_in
if (bigendian) then openr,unit_in,indir+infile,/swap_if_little_endian $
  else openr,unit_in,indir+infile
if (datatype eq 'float') then $
  inassoc=assoc(unit_in,fltarr(nx,ny),headersize)
if (datatype eq 'integer') then $
  inassoc=assoc(unit_in,intarr(nx,ny),headersize)
if (datatype eq 'byte') then $
  inassoc=assoc(unit_in,bytarr(nx,ny),headersize)

; open flow x output file for assoc, write header
get_lun, unit_flox
if (bigendian) then openw,unit_flox,outdir+floxfile,/swap_if_little_endian $
  else openw,unit_flox,outdir+floxfile
floxassoc=assoc(unit_flox,intarr(nx,ny),headersize)
if (headersize ne 0) then begin
  rec=assoc(unit_flox, bytarr(headersize))
  rec[0]=byte(header)
endif

; open flow y output file for assoc, write header
get_lun, unit_floy
if (bigendian) then openw,unit_floy,outdir+floyfile,/swap_if_little_endian $
  else openw,unit_floy,outdir+floyfile
floyassoc=assoc(unit_floy,intarr(nx,ny),headersize)
if (headersize ne 0) then begin
  rec=assoc(unit_floy, bytarr(headersize))
  rec[0]=byte(header)
endif

; open divergence output file for assoc, write header 
get_lun, unit_div
if (bigendian) then openw,unit_div,outdir+divfile,/swap_if_little_endian $
  else openw,unit_div,outdir+divfile
divassoc=assoc(unit_div,intarr(nx,ny),headersize)
if (headersize ne 0) then begin
  rec=assoc(unit_div, bytarr(headersize))
  rec[0]=byte(header)
endif

; open vorticity output file for assoc, write header 
get_lun, unit_vor
if (bigendian) then openw,unit_vor,outdir+vorfile,/swap_if_little_endian $ 
  else openw,unit_vor,outdir+vorfile
vorassoc=assoc(unit_vor,intarr(nx,ny),headersize)
if (headersize ne 0) then begin
  rec=assoc(unit_vor, bytarr(headersize))
  rec[0]=byte(header)
endif

; define array cubeseg 
if (datatype eq 'float') then cubeseg=fltarr(nx,ny,winlength)
if (datatype eq 'integer') then cubeseg=intarr(nx,ny,winlength)
if (datatype eq 'byte') then cubeseg=bytarr(nx,winlength)

; start big loop over timesteps it
for it=itstart,itend do begin 
print,' ==== time step ',it

; define start and end of timecut depending on odd or even length
if (fix(winlength/2) eq winlength/2) then begin
  segstart=limit(fix(it-fix(0.5*winlength)),0,nt-winlength-1)
  segend=segstart+winlength
endif else begin
  segstart=limit(fix(it-floor(0.5*winlength)),0,nt-winlength)
  segend=segstart+winlength-1
endelse

; fill cubeseg for this timestep
for iseg=0,winlength-1 do cubeseg[*,*,iseg]=inassoc[segstart+iseg]

; compute flow map at this timestep
flowvect,cubeseg,vx,vy,fwhm=fwhm

; test vx and vy for NaN values and set these to zero
wnanvx=where(~finite(vx)) ; NaN values in vx?
wnanvy=where(~finite(vy)) ; NaN values in vy?
szwnanvx=size(wnanvx)
if (szwnanvx[0] gt 0) then begin 
  print,'  #### OOPS, number of NaN values in vx: ',szwnanvx[1]
  index=fltarr(nx,ny)       ; where gives 1D vector, need array 
  vx[wnanvx]=index[wnanvx]
endif
szwnanvy=size(wnanvy)
if (szwnanvy[0] gt 0) then begin 
  print,'  #### OOPS, number of NaN values in vy: ',szwnanvy[1]
  index=fltarr(nx,ny)       ; where gives 1D vector, need array 
  vy[wnanvy]=index[wnanvy]
endif

; compute divergence and vorticity maps at this timestep
div=divergence(vx,vy)
vor=vorticity(vx,vy) 

; determine 10^x scale factor fitting integers < 32767 limit
if (it eq itstart) then begin
  vxclip=histo_opt_rr(vx,0.10)  ; heavy clip eg for guiding edges
  clipmom=moment(vxclip,sdev=sdevclip,/double)
  scalefac=10^(4-fix(alog10(clipmom[0]+5*sdevclip)))
  if ((clipmom[0]+5*sdevclip)*scalefac gt 32000) then scalefac=scalefac/10
  if ((clipmom[0]+5*sdevclip)*scalefac gt 32000) then scalefac=scalefac/10
  print,' ==== scalefactor: all outputs multiplied by ',scalefac
endif
   ; NB: subsequently use imin, imax in ximovie for display clipping

; output after multiplication with scalefactor
floxassoc[it-itstart]=fix(vx*scalefac)
floyassoc[it-itstart]=fix(vy*scalefac)
divassoc[it-itstart]=fix(div*scalefac)
vorassoc[it-itstart]=fix(vor*scalefac)

; end big loop over timesteps it
endfor

; free the input and output files
free_lun,unit_in,unit_flox,unit_floy,unit_div,unit_vor

print,' === makeflowcubefiles properly completed'
end

