; file: histo_opt_rr.pro
; init: 06-Jul-1993  Pit Suetterlin, KIS
; last: Mar 27 2018  Rob Rutten  Deil

;+

FUNCTION histo_opt_rr, image, cutoff, ix, $
  top_only=top_only, bottom_only=bottom_only

; NAME:
;       HISTO_OPT
; PURPOSE:
;       Clip image values which are the CUTOFF brightest or darkest,
;       resp. 
; CALLING SEQUENCE:
;       CLIP_IMAGE = HISTO_OPT ( IMAGE [, CUTOFF [, IX]] [,<keywords>])
; INPUTS:
;       IMAGE : Array with data. may be 1 to 3dim
; OPTIONAL PARAMETERS:
;RR  cutoff default 1E-3       
;       IX    : (Output) Contains indices of the clipped values
; KEYWORDS:
;       top_only: (Flag) Clip only the upper values
;       bottom_only: (Flag)  "     "  "   lower   "
; OUTPUTS:
;       image with the CUTOFF fraction lowest and highest values set
;       to the value of the next highest/lowest point.
; RESTRICTIONS:
;       Maybe this should be a procedure, as it uses a lot of memory
;       for big arrays.  However, it is used mainly for displaying, so you
;       wouldn't want to change the original data.
; PROCEDURE:
;       Compute histogram, evaluate the boundaries and return
;       IMAGE>LOW<HIGH 
; MODIFICATION HISTORY:
;       06-Jul-1993 P.Suetterlin, KIS
;       16-Feb-1995 P.Suetterlin, KIS: Take care for float
;                   arrays. Histogram doesn't like them.
;       Mar 27 2018 RR: () > [], _rr = no conversion to long 
;-

on_error, 2

IF n_params() EQ 0 THEN BEGIN
  message, 'Usage: RESULT = HISTO_OPT ( IMAGE [,CUTOFF] )', /cont
  return, undefined
ENDIF

; defaults for keywords
if (n_elements(bottom_only) eq 0) then bottom_only=0
if (n_elements(top_only) eq 0) then top_only=0

; default cutoff
IF (n_params() LT 2) THEN cutoff = 1.e-3

; image size 
s = size(image)
 ;;;
 ;;; If the image is in a float format, then histogram() doesn't know
 ;;; what to do. In that case, convert to fix. But then you have to be
 ;;; sure that the range is OK (especially for normalized images with
 ;;; a range from 0. to 1.). 
 ;;;
IF s[s[0]+1] GT 3 THEN BEGIN
  fak = 10000./(max(image, min = hmin)-hmin)
  h = histogram(fix((image-hmin)*fak))
ENDIF ELSE BEGIN
  h = histogram(image)
  hmin = min(image)
  fak = 1
ENDELSE

nh = n_elements(h)
 ;;;
 ;;; Integrate the histogram so that h(i) holds the number of points
 ;;; with equal or lower intensity.
 ;;;
FOR i = 1l, nh-1 DO h[i] = h[i]+h[i-1]
 ;;;
 ;;; and normalize it to unity
 ;;;
h = float(h)/h[nh-1]
 ;;;
 ;;; As CUTOFF is in percent and h is normalized to unity,
 ;;; cmin/cmax are the indices of the point where the number of pixels
 ;;; with lower/higher intensity reach the given limit. This has to be
 ;;; converted to a real image value by dividing by the scalefactor
 ;;; FAK and adding the min value of the image
 ;;;
cmin = max(where(h LE cutoff))/fak+hmin 
cmax = min(where(h GE (1.-cutoff)))/fak+hmin

;; cmin=float(cmin)
;; cmax=float(cmax)

 ;;;
 ;;; Where is slow. Only compute if requested.
 ;;;
IF n_params() EQ 3 THEN ix = where((image LE cmin) OR (image GE cmax))

;RR "where" gives results in long, cmin, cmax are long, propagate to image
;RR so find the type and convert; remains wrong in Pit's original (on SSW)
;RR hence the _rr extension
if (s[s[0]+1] eq 1) then begin 
  cmin=byte(cmin)
  cmax=byte(cmax)
endif
if (s[s[0]+1] eq 2) then begin 
  cmin=fix(cmin)
  cmax=fix(cmax)
endif
if (s[s[0]+1] eq 4) then begin 
  cmin=float(cmin)
  cmax=float(cmax)
endif

; now cut 
IF (top_only) THEN $
  return, image < cmax $
ELSE IF (bottom_only) THEN $
  return, image > cmin $
ELSE $
  return, image > cmin < cmax

END


; ============= test per IDLWAVE Hyper-C  ========================

cd,'/home/rutten/data/SDO/2014-06-14-small/target/level2'
data=readfits('AIA20140614_090011_0171.fits.fz')
histdat=histo_opt_rr(data,1.E-2,/top_only)
sv,histdat

end
