; file: flowvect.pro
; init: Jan 22 2013 taken from ~/rr/wrk/DOT/2005-10-14/vanveelen-idl
; note: needs sconvol.pro, odd.pro, fivepoint.pro
;       output: vx and vy arrays for divergence.pro, vorticity.pro
;       from Roberto Molowny Horas (ITA Oslo early 1990s) via Pit Suetterlin
;       I (RR) think it adds up a sequence of vectors per subsequent pairs

PRO Flowvect, mov, VX, VY, STD=std, FWHM=fwhm, BOXCAR=boxcar, $
              ADIF=adif, CORR=corr, QFIT2=qfit2, CROSSD=crossd
;+
; NAME:
;	FLOWVECT
;
; PURPOSE:
;	Compute flow maps.
;
; CALLING SEQUENCE:
;	FLOWVECT, ARR, VX, VY, [ STD=, FWHM=, BOXCAR=,
;		ADIF=, CORR=, QFIT2=, CROSSD=]
;
; INPUTS:
;       ARR:   3D array (sx,sy,num) with image data.
;
;       STD = standard deviation of a 2-D gaussian.
;
;       FWHM = full width at half maximum of the 2-D gaussian. For further
;               information, see the header of SCONVOL.
;
;       BOXCAR = width of a boxcar running window. This keyword supersedes
;               STD and FWHM.
;
; KEYWORDS:
;	ADIF = uses an absolute differences algorithm.
;
;	CORR = uses a multiplicative algorithm. Default is the sum of
;		square of the local differences.
;
;	QFIT2 = uses 9 points fitting procedure.
;
;	CROSSD = uses cross derivative interpolation formulae.
;
; OUTPUTS:
;	VX,VY = X and Y components for the proper motion map.
;
; SIDE EFFECTS:
;	None.
;
; COMMON BLOCKS:
;	None.
;
; PROCEDURE:
;	It uses November's method of shifting BOTH images. The defaults for
;	the different methods are square differences for the matching, 
;	gaussian window for the smoothing and FIVEPOINT for the subpixel
;	extrem finding procedure. Images must be in FITS format.
;
; EXAMPLE:
;	Let name1 and name2 be two string arrays containing the names
;	for first and second images of every pair, respectively. For an
;	image resolution of 0".2 per pixel, we want to compute a flow map
;	using a FWHM of 4" for the smoothing gaussian window. Therefore,
;	the call will be:
;
;	IDL> FLOW_MAKER,name1,name2,vx,vy,fwhm=8.5
;
; REFERENCES:
;       * November, L.J., 1986, Applied Optics, Vol.25, No.3, 392
;	* November,L.J. and Simon,G.W.: 1988, Ap.J., 333, 427
;
; MODIFICATION HISTORY:
;	R. Molowny Horas, May 1992.
;	Keywords added in July 1992, RMH
;	FITS routines added in November 1992, RMH
;	Supressed keyword SILENT, May 1994, RMH
;       2000-03-24 Pit Sueterlin Change to using large image array
;-

ON_ERROR, 2

IF n_params() NE 3 THEN message, 'Usage: FLOWVECT, ARR, VX, VY'

s = size(mov)
IF s(0) NE 3 THEN Message, 'First argument should be a 3D array!'
n = s(3)
n_el = s(s(0)+2)

cc = FLTARR(s(1), s(2), 3, 3)   ;The cum. correlation function.

FOR k=0, n-2 DO BEGIN
    a = mov(*, *, k)
    b = mov(*, *, k+1)
    a = a - TOTAL(a)/n_el       ;Remove mean.
    b = b - TOTAL(b)/n_el
    FOR i=-1, 1 DO FOR j=-1, 1 DO CASE 1 OF ;Methods.
        KEYWORD_SET(adif): BEGIN ;Absolute differences.
            cc(0, 0, i+1, j+1) = cc(*, *, i+1, j+1)+$
              ABS(SHIFT(a, i, j)-SHIFT(b, -i, -j))
        END
        KEYWORD_SET(corr): BEGIN ;Cross products.
            cc(0, 0, i+1, j+1) = cc(*, *, i+1, j+1)+$
              SHIFT(a, i, j) * SHIFT(b, -i, -j)
        END
        Else: BEGIN             ;Square differences.
            dumb = SHIFT(a, i, j) - SHIFT(b, -i, -j)
            cc(0, 0, i+1, j+1) = cc(*, *, i+1, j+1) + dumb*dumb
            dumb = 0            ;This should be faster than (...)^2
        END
    ENDCASE
    a = 0 & b=0
ENDFOR

cc(0, 0, 0, 0) = cc(1, *, *, *) ;Takes care of the edges.
cc(0, 0, 0, 0) = cc(*, 1, *, *)
cc(s(1)-1, 0, 0, 0) = cc(s(1)-2, *, *, *)
cc(0, s(2)-1, 0, 0) = cc(*, s(2)-2, *, *)

FOR i=0, 2 DO FOR j=0, 2 DO IF N_ELEMENTS(boxcar) NE 0 THEN $
  cc(0, 0, i, j)=KCONVOL(cc(*, *, i, j), boxcar) ELSE $	;Boxcar...
  cc(0, 0, i, j)=SCONVOL(cc(*, *, i, j), std=std, fwhm=fwhm) ;or gausian.

CASE 1 OF
    KEYWORD_SET(qfit2): QFIT2, cc, vx, vy ;9-p. fitting,
    KEYWORD_SET(crossd): CROSSD, cc, vx, vy ;cross derivat.
    Else: FIVEPOINT, cc, vx, vy ;or default.
ENDCASE

vx = 2. * vx & vy=2. * vy       ;Scales the result.

END
