PRO enhance, mov, ASSOC_SIZE=as, BEST=best, plotout=plotout
;+
; NAME:
;       ENHANCE
; PURPOSE:
;       Powerspectrum assimilation for a series of images
; CALLING SEQUENCE:
;       Enhance, Data
; INPUTS:
;       Data: 3D array of image data, typically a movie time series
; SIDE EFFECTS:
;       Data is overwritten with the result!
; RESTRICTIONS:
;       All images of the series must show basically the same object
; PROCEDURE:
;       Find best-RMS image; compute the azimuthally averaged power
;       spectra (APS) for all images; define (for each image) a power
;       enhancement function based on the ratio of the APS of current
;       and best image.  enhance FFt amplitudes.
; MODIFICATION HISTORY:
;       12-Apr-2001  P.Suetterlin, SIU
;       22-Jan-2002  assoc arrays possible;  check dimensions
;       05-Mar-2002  catch division by zero; don't change average
;                    intensity. 
;       20-Jan-2004  check the enhanced array against overruns
;       21-Oct-2005  Use continue or out of range if best=last
;       Dec 10 2010 RR: add plotout keyword to switch off screen plots
;-

  ;;; Trick around with the reconstructions:
  ;;; Get all to have the powerspectrum of the best

  ;;; When mov is an assoc array, the variable as has to hold the
  ;;; number of images.  Else use size to find it.

IF keyword_set(as) THEN BEGIN
    s = size(mov)
    num = as
ENDIF ELSE BEGIN
    s = size(mov)
    IF s(0) NE 3 THEN BEGIN
        message, 'Only for processing 3-D data cubes.  Aborting.', /info
        return
    ENDIF
    num = s(3)
ENDELSE

CASE s(s(0)+1) OF
    1: limit = [0, 255]
    2: limit = [-32767, 32767]
ENDCASE

;RR addition
if (n_elements(plotout) eq 0) then plotout=0

;RR
if (plotout) then begin
  window, 0
  window, 1, xs=s(1), ys=s(2)
endif

IF NOT keyword_set(best) THEN BEGIN
     ;;; find best RMS image
    rms = fltarr(num)
    FOR i=0, num-1 DO rms(i) = stdev(mov(*, *, i), m)/m
    mx = max(rms, best)
ENDIF

  ;;; compute reference powerspectrum

fp = fft(mov(*, *, best), -1)
fpa= abs(fp)
fp = fp/fpa
pi = polarint(fpa)

x = dist(s(1), s(2))

  ;;; now for all images
FOR img=0, num-1 DO BEGIN
    IF img EQ best THEN CONTINUE
    fp1 = fft(mov(*, *, img), -1)
    fpa1 = abs(fp1)
    fp1 = fp1/fpa1
      ;;; careful with zero values!
    IF min(fpa1) EQ 0 THEN fp1(where(fpa1 EQ 0)) = 0
    pi1 = polarint(fpa1)
    if (plotout) then begin
      wset, 0
      plot_io, pi
      oplot, pi1
    endif
    pq = smoother(pi/pi1)
    q = interpolate(pq, x, missing=1)
      ;;; average mustn't be changed
    q(0) = 1.
    
    p1 = float(fft(q*fpa1*fp1, 1))
    
    IF keyword_set(limit) THEN p1 = p1 > limit(0) < limit(1)
    
    IF keyword_set(as) THEN $
      mov(img)=p1 $
    ELSE $
      mov(*, *, img) = p1
    if (plotout) then begin
      wset, 1
      tvscl, p1
      xyouts, 10, 10, /dev, string(img+1, num, form="(i3,' of ',i3)")
      empty
    endif
ENDFOR

END
