pro bandec,a,al,indx,m1,m2,d
;+
; NAME:
;	BANDEC
; PURPOSE:
;	Compute the LU-Decomposition of a bandmatrix
; CATEGORY:
;	Matrix operations, solution of linear equation systems
; CALLING SEQUENCE:
;	BANDEC,A,AL,INDX,M1,M2[,D]
; INPUTS:
;	A   : Array (N,M1+M2+1) with the bandmatrix. Storage sheme see
;             below.  **Changed by the program**
;       M1  : number of subdiagonals of a
;       M2  : number of superdiagonals of a
; OUTPUTS:
; 	A   : Upper triangular matrix of the LU decomposition,DIM(N,M1+M2+1
;       AL  : Lower    "         "    "   "   "    "         ,DIM(N,M1)
;       INDX: Vector of length n recording the row permutations from
;             pivoting.
;       D   : +- 1 depending on the number of permutations: parity
; SIDE EFFECTS:
; 	Matrix a is altered. If you want to keep it for later use, make
;       a copy before calling bandec.
; RESTRICTIONS:
; 	
; PROCEDURE:
; 	For a complete description see 
;          Press et al.  Numerical recipes - The art of scientific computing
;       Storage sheme for a:
;       A holds the non-zero elements of a square bandmatrix (from nowon
;       referred as A0). The dimension of A is ( N , M1 + M2 + 1) with
;       N,M1 and M2 as defined above. The diagonal elements of A0 are
;       stored in A(*,M1), the subdiagonal Elements in A(*,0:M1-1) and
;       the superdiagonal elements in A(*,M1+1:M1+M2). Nondefined
;       Elements are set to zero, so the first rows for a bandmatrix with
;       N=10,M1=2,M2=3 will look like this:
;       
;       [ 0      , 0      , A0(0,0), A0(1,0), A0(2,0), A0(3,0) ]
;       [ 0      , A0(0,1), A0(1,1), A0(2,1), A0(3,1), A0(4,1) ]
;       [ A0(0,2), A0(1,2), A0(2,2), A0(3,2), A0(4,2), A0(5,2) ]
;       [ A0(1,3), A0(2,3), A0(3,3), A0(4,3), A0(5,3), A0(6,3) ]
;            .        .        .        .        .      .
;            .        .        .        .        .      .
;       etc.  Last line would be
;
;       [ A0(7,9), A0(8,9), A0(9,9), 0      , 0      , 0       ]
;       
; MODIFICATION HISTORY:
;	28-02-1993  PS, adapted from Numerical Recipes in C
;-

on_error,2             ;;;  Return to caller

;;;
;;;  Some error checking...
;;;

if n_params() lt 5 then $
  message,'Usage:  bandec,a,al,indx,m1,m2[,d]'
sa=size(a)
if sa(0) ne 2 then message,'Must be a matrix: A'
n=sa(1)
if (m1+m2+1) ne sa(2) then message,'Check parameters: M1,M2'

for i=0,m1-1 do a(i,*)=shift(a(i,*),i-m1)

d=1.0
l=m1-1
nm1=n-1
for k=0,nm1 do begin
  dum=a(k,0)
  i=k
  if (l lt nm1) then l=l+1
  for j=k+1,l do begin
    if abs(a(j,0)) gt abs(dum) then begin
      dum=a(j,0)
      i=j
    endif
  endfor
  indx(k)=i
  if (dum eq 0) then a(k,0)=1e-20
  if (i ne k) then begin
    d=-d
    dum=a(k,*)
    a(k,*)=a(i,*)
    a(i,*)=dum
  endif
  for i=k+1,l do begin
    dum=a(i,0)/a(k,0)
    al(k,i-k-1)=dum
    a(i,*)=[[a(i,1:*)-dum*a(k,1:*)],[0.]]
  endfor
endfor
end
