; file: scatcont_averdelay.pro
; init: Apr 11 2018  Rob Rutten  Deil
; last: Jul 30 2019  Rob Rutten  Deil

;+

pro scatcont_averdelay,fileA,fileB,psstrA,psstrB,$
  itselfile=itselfile,itdelay=itdelay,smeareach=smeareach,$
  xrange=xrange,yrange=yrange,trange=trange,tskip=tskip,$
  cadence=cadence,imagemask=imagemask,$
; ----- options for scatcont.pro
  plotrangeA=plotrangeA,plotrangeB=plotrangeB,$
  boxrangeA=boxrangeA,boxrangeB=boxrangeB,$
  boxcolor=boxcolor,boxthick=boxthick,binscale=binscale,$
  outerlevel=outerlevel,nbins=nbins,contourstep=contourstep,$
  nocloud=nocloud,nomoments=nomoments,nohistograms=nohistograms,$
  nosummit=nosummit,nonumber=nonumber,nostep=nostep,$
  fullpearson=fullpearson,quadpearson=quadpearson,boxpearson=boxpearson,$
  smear=smear,label=label,fontsize=fontsize,thick=thick,$
  scaleA=scaleA,scaleB=scaleB,$
; ----- options for plotoutput
  extralabel=extralabel,psdir=psdir,$
  xaxtitle=xaxtitle,yaxtitle=yaxtitle,$
  showgv=showgv,_extra=plotkeywords

 ; NAME:
 ;   scatcont_averdelay
 ;   
 ; PURPOSE:
 ;   use scatcont.pro on fits files sampling multi-frame selection with delay
 ;   
 ; DESCRIPTION:
 ;   see comments in program and example at file bottom
 ;   
 ; CALL:
 ;   see above
 ;   
 ; INPUTS:
 ;   fileA = string path+name (x,y,t) fits file A
 ;   fileB = string path+name (x,y,t) fits file B
 ;   psstrA = string defining quantity A for ps filename
 ;   psstrB = string defining quantity B for ps filename
 ; 
 ; OPTIONAL KEYWORD INPUTS:
 ;   itselfile: ascii file with selected it values (e.g., best rms)
 ;   itdelay: delay to sample B after A in time steps
 ;   smeareach: nr of px to smear each; negative = also rebin (default 0)
 ;   xrange, yrange, trange: 2-elem delimiter specifications
 ;   tskip = range of time steps to be skipped for speed
 ;   cadence: in sec, only used to give delay in minutes in plotlabel
 ;   imagemask: apply image mask = 1/0 array (x,y) with 1 = pass this pixel
 ;     (replaces older scatcont block parameter)
 ;   scaleA, scaleB: factors to divide by to get nicer axis numbers
 ; ---- options for scatcont.pro: see there
 ;   xaxtitle, yaxtitle: strings for axis labels
 ;   extralabel=2/1/0: lots, only t_delay, none; or string to be copied
 ;   showgv=1/0: show resulting plot
 ; 
 ; OUTPUTS:
 ;   scatcont plot
 ;   
 ; HISTORY:
 ;   Apr 12 2018 RR: start from movex_scatcont.pro
 ;   May 15 2019 RR: imagemask
;-

; answer no-parameter query 
if (n_params(0) lt 4) then begin
  sp,scatcont_averdelay
  return
endif

; defaults for keywords
if (n_elements(itselfile) eq 0) then itselfile=''
if (n_elements(itdelay) eq 0) then itdelay=0
if (n_elements(smeareach) eq 0) then smeareach=0
if (n_elements(xrange) eq 0) then xrange=[0,-1]
if (n_elements(yrange) eq 0) then yrange=[0,-1]
if (n_elements(trange) eq 0) then trange=[0,-1]
if (n_elements(tskip) eq 0) then tskip=[-1,-1]
if (n_elements(cadence) eq 0) then cadence=1
if (n_elements(imagemask) eq 0) then imagemask=-1
if (n_elements(xaxtitle) eq 0) then xaxtitle='A'
if (n_elements(yaxtitle) eq 0) then yaxtitle='B'
if (n_elements(scaleA) eq 0) then scaleA=1
if (n_elements(scaleB) eq 0) then scaleB=1
if (n_elements(label) eq 0) then label=''
if (n_elements(extralabel) eq 0) then extralabel=0
if (n_elements(psdir) eq 0) then psdir='/tmp'
if (n_elements(showgv) eq 0) then showgv=0

; open files A and B for assoc (must be single-stream fits files)
fits_openassoc,fileA,1,nx,ny,nt,bitpixA,headeroffA,swap_endianA
fits_openassoc,fileB,2,nxB,nyB,ntB,bitpixB,headeroffB,swap_endianB

; check size equality
if (nx ne nxB or ny ne nyB or nt ne ntB) then begin
  print,' ##### scatcont_averdelay abort: files A and B inequal sizes'
  return
endif

; xrange and yrange 
xmin=xrange[0]
xmax=xrange[1]
ymin=yrange[0]
ymax=yrange[1]
if (xmax eq -1) then xmax=nx-1
if (ymax eq -1) then ymax=ny-1
nxcut=xmax-xmin+1
nycut=ymax-ymin+1

; trange
itstart=trange[0]
itend=trange[1]
if (itend eq -1) then itend=nt-1 
ntcut=itend-itstart+1

; later start for negative itdelay
if (itdelay lt 0 and trange[0]+itdelay lt 0) then itstart=-itdelay

; define assocs
if (bitpixA eq 8) then assarrA=bytarr(nx,ny)
if (bitpixA eq 16) then assarrA=intarr(nx,ny)
if (bitpixA eq -32) then assarrA=fltarr(nx,ny)
inassocA=assoc(1,assarrA,headeroffA)
if (bitpixB eq 8) then assarrB=bytarr(nx,ny)
if (bitpixB eq 16) then assarrB=intarr(nx,ny)
if (bitpixB eq -32) then assarrB=fltarr(nx,ny)
inassocB=assoc(2,assarrB,headeroffB)

; get optional rms-screened it array
if (itselfile ne '') then readcol,itselfile,it_sel,format='I' $
else it_sel=indgen(nt)

; ------ loop over it range

nritsteps=0
firstit=1
for it=itstart,itend do begin

; skip segment of sequence if within tskip range
  if (it ge tskip[0] and it le tskip[1]) then goto, SKIPTHISIT 

; don't go per delay beyond end of file
  if (it+itdelay eq nt) then begin
    itend=it-1
    break
  endif

; don't use this it if not in itselfile and increase itstart if needed
  if (min(abs(it_sel-it)) gt 0.1) then begin
    if (it eq itstart) then itstart=itstart+1
    goto, SKIPTHISIT 
  endif

; same if it+itdelay not in itselfile
  if (min(abs(it_sel-(it+itdelay))) gt 0.1) then begin
    if (it eq itstart) then itstart=itstart+1
    goto, SKIPTHISIT 
  endif

; add to nritsteps
  nritsteps=nritsteps+1

; get images
  imA=inassocA[it]
  imB=inassocB[it+itdelay]
  imA=imA[xmin:xmax,ymin:ymax]
  imB=imB[xmin:xmax,ymin:ymax]

; optional smear and optional rebin (both)
  if (smeareach gt 1) then begin
    smearthis=smeareach
    imA=smooth(imA,smearthis,/edge_truncate)
    imB=smooth(imB,smearthis,/edge_truncate)
  endif
  if (smeareach lt 0) then begin
    smearthis=abs(smeareach)
    imA=imA[0:fix(nxcut/smearthis)*smearthis-1,$
            0:fix(nycut/smearthis)*smearthis-1]
    imB=imB[0:fix(nxcut/smearthis)*smearthis-1,$
            0:fix(nycut/smearthis)*smearthis-1]
    imA=rebin(imA,fix(nxcut/smearthis),fix(nycut/smearthis))
    imB=rebin(imB,fix(nxcut/smearthis),fix(nycut/smearthis))
  endif

; optionally mask part of images
  if (imagemask[0] ne -1) then begin
    imA=imA[where(imagemask eq 1)]
    imB=imB[where(imagemask eq 1)]
  endif
  
; store in large data arrays for scatcont
  if (firstit) then begin
    dataA=imA
    dataB=imB
    firstit=0
  endif else begin
    dataA=[dataA,imA] ; side-by-side concatenation, first dim = nx*nt
    dataB=[dataB,imB]
  endelse

SKIPTHISIT:
endfor ; end of it loop over time steps

; get sample size for extralabel
szim=size(imA)
nxim=szim[1]
nyim=szim[2]
npairs=float(nxim)*nyim*nritsteps

; ===== set up and run scatcont

; filename (mucklast = make delay < 0 to get proper-order file name)
if (itdelay lt 0) then mucklast=itend-1 else mucklast=itend
if (smeareach ne 0) then $
  sm='_sm'+strmid(string(smearthis+1E2,format='(I3)'),1)+'_' else sm='_'
if (tskip[0] ne -1) then skip='skip_'+trim(tskip[0])+'_'+trim(tskip[1])+'_' $
else skip=''
psfile=psdir+'/'+'scat_'+trim(psstrA)+'_'+trim(psstrB)+sm+skip+$
  ;; trim(xmin)+'_'+trim(xmax)+'_'+$
  ;; trim(ymin)+'_'+trim(ymax)+'_'+$
  ;; strmid(string(itstart+1E3,format='(f5.0)'),1,3)+'_'+$
  ;; strmid(string(mucklast+1E3,format='(f5.0)'),1,3)+'_'+$
  ;; strmid(string(nritsteps+1E3,format='(f5.0)'),1,3)+'_'+$
  strmid(string(itdelay+1E4,format='(f6.0)'),1,3)+'.ps'

; label = given in extralabel or default choice
szlabel=size(extralabel)
if (szlabel[szlabel[0]+1] eq 7) then label=extralabel else begin
  if (extralabel gt 0) then if (cadence eq 0) then $
    ;; label='delay '+trim(itdelay)+' time steps' else $
    ;;   label='delay = '+trim(itdelay*cadence/60.,'(F6.1)')+' min'
    label=[label,trim(itdelay)+' time steps'] else $
      label=[label,trim(itdelay*cadence/60.,'(F6.1)')+' min']
  if (extralabel gt 1) then begin
    if (smeareach ne 0) then label=[label,'smeareach = '+trim(smeareach)]
    label=[label,'x_range = ['+trim(xmin)+','+trim(xmax)+']']
    label=[label,'y_range = ['+trim(ymin)+','+trim(ymax)+']']
    label=[label,'t_range = ['+trim(itstart)+','+trim(itend)+']']
    if (tskip[0] ne -1) then $
      label=[label,'t_skip =  ['+trim(tskip[0])+','+trim(tskip[1])+']']
    label=[label,'N_pairs = '+trim(npairs,'(E8.1)')]
  endif
endelse

; call scatcont
scatcont,dataA,dataB,blockbox=[0,0,0,0],$
  plotrangeA=plotrangeA,plotrangeB=plotrangeB,$
  boxrangeA=boxrangeA,boxrangeB=boxrangeB,$
  boxcolor=boxcolor,boxthick=boxthick,binscale=binscale,$
  outerlevel=outerlevel,nbins=nbins,contourstep=contourstep,$
  nocloud=nocloud,nomoments=nomoments,nohistograms=nohistograms,$
  nosummit=nosummit,nonumber=nonumber,nostep=nostep,$
  fullpearson=fullpearson,quadpearson=quadpearson,boxpearson=boxpearson,$
  smear=smear,label=label,psfile=psfile,$
  fontsize=fontsize,thick=thick,xtitle=xaxtitle,ytitle=yaxtitle,$
  scaleA=scaleA,scaleB=scaleB

; ----- clean and exit

; close lun's
free_lun,1
free_lun,2

; optional showing
if (showgv) then spawn,'gv '+psfile

;  print time of job completion (for large ones, e.g., in UU)
spawn,'date'  

end

; =============== main for testing per IDLWAVE H-c ======================

cd,'/home/rutten/data/SST/2014-06-21-quiet'

cadence=11.564974
smeareach=-10
label='data identifier'
extralabel=1
showgv=1
psdir='/tmp'
itselfile='sst_iris_sdo/rms_itsel.txt'

nocloud=0
outerlevel=20
contourstep=5

;; ; select whether contrail area only or full-frame (comment out)
;; xrange=[250,750]    ; contrail area only
;; yrange=[250,750]
;; trange=[75,95]      ; contrail precursor duration
;; ;;;; tskip=[80,100] ; skip the period of the precursor
;; trange=[88,88]  ; only 1 time step = precursor
;; trange=[50,150]

; setup A
fileA='constructs/comb_HaW+CaI.fits'
fileA='constructs/comb_HaW-CaI.fits'
psstrA='HaW+CaI'
psstrA='HaW-CaI'
xaxtitle='HaW + CaI'
xaxtitle='HaW - CaI'
;;;; plotrangeA=[1400,2990] ; nonsmear
;;;; plotrangeA=[-600,600] ; nonsmear
;; plotrangeA=[1500,2790]
;; plotrangeA=[-500,500]
scaleA=1.

; setup B
fileB='sst_iris_sdo/iris_SJI_1400_SST.fits'
psstrB='IRIS1400'   
yaxtitle='IRIS 1400'   ; OOPS hoe AA?
;;;; plotrangeB=[-3000,3.0E4] ; nonsmear
scaleB=1.
;; plotrangeB=[300,2.1E3] 

; set time delay sampling
;; for itdelay=-75,75 do begin  ; full movie (SLOW and eats laptop!) 
;; for itdelay=-16,16,16 do begin
;; for itdelay=-4,4,4 do begin
;; for itdelay=-20,20,20 do begin  
for itdelay=0,0 do begin  

  scatcont_averdelay,fileA,fileB,psstrA,psstrB,$
    itselfile=itselfile,itdelay=itdelay,smeareach=smeareach,$
    xrange=xrange,yrange=yrange,trange=trange,tskip=tskip,$
    cadence=cadence,imagemask=imagemask,$
; ----- options for scatcont 
    plotrangeA=plotrangeA,plotrangeB=plotrangeB,$
    boxrangeA=boxrangeA,boxrangeB=boxrangeB,$
    boxcolor=boxcolor,boxthick=boxthick,binscale=binscale,$
    outerlevel=outerlevel,nbins=nbins,contourstep=contourstep,$
    nocloud=nocloud,nomoments=nomoments,nohistograms=nohistograms,$
    nosummit=nosummit,nonumber=nonumber,nostep=nostep,$
    fullpearson=fullpearson,quadpearson=quadpearson,boxpearson=boxpearson,$
    label=label,$
; ----- options for plotoutput here
    extralabel=extralabel,psdir=psdir,$
    xaxtitle=xaxtitle,yaxtitle=yaxtitle,showgv=showgv

endfor

;; ; ps > png
;; spawn,'cd '+psdir+'; /home/rutten/rr/bin/pscroppngall'

end
