; file: scatcont.pro
; init: Apr 30 2015  Rob Rutten  Deil from Alfred de Wijn's version
; last: Aug 29 2020  Rob Rutten  Deil
; note: main part under MAIN below
; todo: maskimage as in scatcont_averdelay.pro

;+
; NAME: scatcont.pro
;
; PURPOSE: plot px-by-px scatter diagram in Strous format = replace
;   sample points by point density contours to avoid saturation
;   (Louis Strous: Figs. 13.2-13.3 of 1994PhDT.......347S = his thesis)
;
; METHOD:
;   based on Alfred de Wijn's version
;
; CALL (in MAIN part below):
;   scatcont,dataA,dataB,$
;     blockbox=blockbox,plotrangeA=plotrangeA,plotrangeB=plotrangeB,$
;     boxrangeA=boxrangeA,boxrangeB=boxrangeB,$
;     boxcolor=boxcolor,boxthick=boxthick,binscale=binscale,$
;     outerlevel=outerlevel,nbins=nbins,contourstep=contourstep,$
;     nocloud=nocloud,nomoments=nomoments,nohistograms=nohistograms,$
;     nosummit=nosummit,nonumber=nonumber,nostep=nostep,$
;     fullpearson=fullpearson,quadpearson=quadpearson,boxpearson=boxpearson,$
;     smear=smear,label=label,psfile=psfile,$
;     fontsize=fontsize,thick=thick,scaleA=scaleA,scaleB=scaleB,
;     _extra=plotkeywords]
;
; INPUTS:
;   dataA = input array (1D,2D,3D)
;   dataB = input array (1D,2D,3D) with same dimensions as dataA
;
; WARNINGS:
;   some parameters are sticky, run new IDL session to refresh
; 
; OPTIONAL KEYWORD INPUTS:
;   blockbox = [x1,x2,y1,y2,t1,t2] discard subset  ([0,0] passes full axis)
;   plotrangeA = x-axis range, default minmax(dataA)+extensions
;   plotrangeB = y-axis range, default minmax(dataB)+extensions
;   boxrangeA = range of colored overlay box(es) in x direction (default none)
;   boxrangeB = range of colored overlay box(es) in y direction (default none)
;     (multiple boxes: [2,nbox] e.g.: boxrangeA=[[20,40],[40,60],[60,80]])
;   boxcolor =  color of box(es), default cgcolor('green') (strarr[nbox]) 
;   boxthick = thickness of overlaybox frame (default 2)
;   binscale 1/0: rescale nbins from plot range to data range
;   outerlevel = point density at outer level (default 20)
;   nbins = number of bins in data range for contour finding (default 50)
;   contourstep = density increase factor between contours (default 2)
;   nocloud 1/0: no outer sample cloud outside (default 0 = add)
;   nomoments 1/0: no dashed first moment x and y curves (default 0 = add)
;   nohistograms 1/0: no occurrence histogram curves (default 0 = add)
;   nosummit 1.0: no cross at mountain summit (default 0 = add)
;   nonumber = 1/0: no specification number nr pixel pairs (defaul 0 = add)
;   nostep = 1/0: no specification contour step factor (default 0 = add)
;   fullpearson = 1/0: add overall Pearson correlation (default 0)
;   quadpearson = 1/0: add quadrant Pearson correlations (default 0)
;   boxpearson = 1/0 add Pearson for box(es - e.g. [0,0,0,1] for 4 boxes)
;   smear = nr of px to smear over; if negative then rebin (default 0 = none)
;   label = string or string array to insert plot labels upper-lef
;   psfile = string path/filename for output as ps file instead of on screen 
;   fontsize = character size (default 9)
;   thick = line and axis thickness (default 2)
;   scaleA, scaleB: factors to divide by to get nicer (smaller) axis numbers
;   additional: all IDL plot keywords
;
; WARNING:
;   parameters as colors are sticky; restart IDL session after changing them
;
; MODIFICATION HISTORY: 
;   1996 RR: suggested by Louis Strous re Fig. 6 1995ESASP.376a.151R
;   1997 Roar Skartlien (Oslo): better contours
;   1997 Yves Gallant and Mandy Hagenaar (Utrecht): various rewrites
;   2001 RR: add distribution curves
;   2002 Thijs Krijger (Utrecht): reworked version 
;   Aug  3 2010 RR: arrays (..) > [..] and cleanup
;   Apr 30 2015 RR: switch to Alfred de Wijn's scatcont_adw.pro
;   Jul 13 2017 RR: from ps to screen or ps, rename to scatcont.pro
;   Apr 12 2018 RR: Pearson, smear, cloud, scale
;   Apr 25 2018 RR: overlay color box
;   Jul 20 2018 RR: multiple boxes
;   Jul 27 2019 RR: summit cross, quad-Pearsons, binscale, blockbox
;-

;-----------------------------------------------------------------------
function scfunc,p
;RR determines intersection of the two moment curves
	common scattercommon, xmomval,xmomind,ymomind,ymomval
	xmvint = interpol(xmomval,xmomind,p[0])
	ymvint = interpol(ymomval,ymomind,p[1])
	return,sqrt((xmvint-p[1])^2+(ymvint-p[0])^2)
end

;-------------------------------------------------------------------------
; OPLOT_DL.PRO  v1.50  (26 August 2007) --- (c) 1999-2007  H.W. de Wijn
; macro to draw dashed curve   
;RR  Harold W. de Wijn (physicist) is the father of Alfred G. de Wijn
;-------------------------------------------------------------------------
; Parameters:
;   x,y      = supporting points
;   length_1 = length of dash in x-axis data units
;   length_2 = length of blank in x-axis data units
; Keywords:
;   /y_units -> use y-axis instead of x-axis data units
;   /quiet   -> quiet operation
; Other keywords are passed on to OPLOT, which draws the dashes.  Keywords
; supported include: thick, clip, and noclip.  Note that psym is ignored.
;-------------------------------------------------------------------------
PRO OPLOT_DL,x,y,length_1,length_2,y_units=YS,quiet=QU,psym=p,_extra=oplotkw
; occurence of keyword psym ensures its absence in _extra=oplotkw
; psym is set to zero in calls of oplot below
IF n_elements(QU) eq 0 THEN QU=0
ars_y=FLOAT((!y.crange[1]-!y.crange[0])/(!x.crange[1]-!x.crange[0])$
            *(!d.x_size*(!x.window[1]-!x.window[0]))$
            /(!d.y_size*(!y.window[1]-!y.window[0]))) 
;aspect ratio in x,y data units
ars_x=1.0
IF n_elements(YS) eq 0 THEN YS=0
IF YS EQ 1 THEN BEGIN
  ars_x=1.0/ars_y
  ars_y=1.0
ENDIF

size_x=size(x)  &  i_max=size_x[1]-1
IF QU NE 1 THEN print,'OPLOT_DL:  # of supporting points =',i_max+1

xa=x[0]  &  ya=y[0]               ;starting values
i=1l
BEGIN_LOEP:
L=0  &  i=i-1
iaa=i  &  xaa=xa  &  yaa=ya     ;starting values of a dash plus blank

 ;----- dash ----
REPEAT BEGIN                    ;loop until length_1 is overshot
  xx=xa  &  yy=ya               ;starting values of step
  i=i+1                         ;see for which i we surpass length_1
  IF i gt i_max THEN goto,END_DASH
  xxx=x[i]  &  yyy=y[i]         ;end values of step
  xa=xxx  &  ya=yyy             ;starting values of next step
  DL=sqrt(((xxx-xx)/ars_x)^2+((yyy-yy)/ars_y)^2)
  L=L+DL
ENDREP UNTIL (L gt length_1)

xa=xxx-(L-length_1)/DL*(xxx-xx) ;shift back by excess part of last step
ya=yyy-(L-length_1)/DL*(yyy-yy)

IF iaa+1 le i-1 THEN $
  oplot,[xaa,x[iaa+1:i-1],xa],[yaa,y[iaa+1:i-1],ya],psym=0,_extra=oplotkw $
ELSE $
  oplot,[xaa,xa],[yaa,ya],psym=0,_extra=oplotkw

  ;----- blank -----
L=0  &  i=i-1
REPEAT BEGIN                    ;loop until length_2 is overshot
  xx=xa  &  yy=ya               ;starting values of step
  i=i+1                         ;see for which i we surpass length_2
  IF i gt i_max THEN goto,END_LOEP
  xxx=x[i]  &  yyy=y[i]         ;end values of step
  xa=xxx  &  ya=yyy             ;starting values of next step
  DL=sqrt(((xxx-xx)/ars_x)^2+((yyy-yy)/ars_y)^2)
  L=L+DL
ENDREP UNTIL (L gt length_2)

xa=xxx-(L-length_2)/DL*(xxx-xx) ;shift back by excess part of last step
ya=yyy-(L-length_2)/DL*(yyy-yy)

GOTO,BEGIN_LOEP

  ;----- last dash -----
END_DASH: oplot,[xaa,x[iaa+1:i_max]],[yaa,y[iaa+1:i_max]],psym=0,_extra=oplotkw

END_LOEP:
END

; ----------------------- MAIN program -------------------------

pro scatcont,dataA,dataB,$
  blockbox=blockbox,plotrangeA=plotrangeA,plotrangeB=plotrangeB,$
  boxrangeA=boxrangeA,boxrangeB=boxrangeB,$
  boxcolor=boxcolor,boxthick=boxthick,binscale=binscale,$
  outerlevel=outerlevel,nbins=nbins,contourstep=contourstep,$
  nocloud=nocloud,nomoments=nomoments,nohistograms=nohistograms,$
  nosummit=nosummit,nonumber=nonumber,nostep=nostep,$
  fullpearson=fullpearson,quadpearson=quadpearson,boxpearson=boxpearson,$
  smear=smear,label=label,psfile=psfile,$
  fontsize=fontsize,thick=thick,scaleA=scaleA,scaleB=scaleB,$
  _extra=plotkeywords

; keyword defaults
if (n_elements(blockbox) eq 0) then blockbox=[0,0,0,0,0,0]
if (n_elements(plotrangeA) eq 0) then plotrangeA=[0,0]
if (n_elements(plotrangeB) eq 0) then plotrangeB=[0,0]
if (n_elements(boxrangeA) eq 0) then boxrangeA=[0,0]
if (n_elements(boxrangeB) eq 0) then boxrangeB=[0,0]
if (n_elements(boxcolor) eq 0) then boxcolor=-1
if (n_elements(boxthick) eq 0) then boxthick=2
if (n_elements(binscale) eq 0) then binscale=0
if (n_elements(outerlevel) eq 0) then outerlevel=20
if (n_elements(nbins) eq 0) then nbins=50
if (n_elements(contourstep) eq 0) then contourstep=2
if (n_elements(nocloud) eq 0) then nocloud=0
if (n_elements(nomoments) eq 0) then nomoments=0
if (n_elements(nohistograms) eq 0) then nohistograms=0
if (n_elements(nosummit) eq 0) then nosummit=0
if (n_elements(nonumber) eq 0) then nonumber=0
if (n_elements(nostep) eq 0) then nostep=0
if (n_elements(fullpearson) eq 0) then fullpearson=0
if (n_elements(quadpearson) eq 0) then quadpearson=0
if (n_elements(boxpearson) eq 0) then boxpearson=0
if (n_elements(smear) eq 0) then smear=0
if (n_elements(label) eq 0) then label=''
if (n_elements(psfile) eq 0) then psfile=''
if (n_elements(fontsize) eq 0) then fontsize=9
if (n_elements(thick) eq 0) then thick=2
if (n_elements(scaleA) eq 0) then scaleA=1
if (n_elements(scaleB) eq 0) then scaleB=1

; checks

if (contourstep le 1) then begin
  print,' ##### scatcont abort: contourstep must be > 1'
  retall
endif

; ps output 
if (psfile ne '') then begin
  psfilename=psfile   ; use .ps, use pscropepsone/all later 
  openpsplot,psfilename,thick=thick,fontsize=fontsize,xsize=8.8,ysize=8.8
endif 

; optional blocking of a subset
; WARNING !!! revamps data in place, sticks and stinks
if (total(blockbox) gt 0) then begin
  sizedat=size(dataA)
  typedat=sizedat[sizedat[0]+1]
  nxdat=sizedat[1]
  nydat=sizedat[2]
  if (sizedat[0] eq 3) then ntdat=sizedat[3] else ntdat=1
  ; check blockbox values are within data ranges
  if ((blockbox[0] lt 0) or (blockbox[2] lt 0) or (blockbox[4] lt 0) or $
      (blockbox[1] gt nxdat-1) or (blockbox[3] gt nydat-1) $
      or (blockbox[5] gt ntdat-1)) then begin
    print,' ##### scatcont blockbox abort: blockbox exceeds image dimensions'
    retall
  endif
  ; 0,0 option for full axis passing
  if (blockbox[1] eq 0) then blockbox[1]=nxdat-1
  if (blockbox[3] eq 0) then blockbox[3]=nydat-1
  if (blockbox[5] eq 0) then blockbox[5]=ntdat-1
  mask=dataA*0
  if (ntdat eq 1) then mask[blockbox[0]:blockbox[1],$
                            blockbox[2]:blockbox[3]]=1 $
  else mask[blockbox[0]:blockbox[1],$
            blockbox[2]:blockbox[3],$
            blockbox[4]:blockbox[5]]=1 
  ; reform passed into linear arrays
  dataA=dataA[where(mask eq 0)]
  dataB=dataB[where(mask eq 0)]
  dataA=reform(dataA)
  dataB=reform(dataB)
endif

; optional rescale
; WARNING !!! revamps data in place: sticks and stinks
if (scaleA ne 1) then dataA=dataA/float(scaleA)
if (scaleB ne 1) then dataB=dataB/float(scaleB)

; optional smear with optional rebin
if (smear gt 1) then begin
  dataA=smooth(dataA,smear,/edge_truncate)
  dataB=smooth(dataB,smear,/edge_truncate)
endif
if (smear lt 0) then begin
  smear=abs(smear)
  dataA=dataA[0:fix(nxcut/smear)*smear-1,$
              0:fix(nycut/smear)*smear-1]
  dataB=dataB[0:fix(nxcut/smear)*smear-1,$
              0:fix(nycut/smear)*smear-1]
  dataA=rebin(dataA,fix(nxcut/smear),fix(nycut/smear))
  dataB=rebin(dataB,fix(nxcut/smear),fix(nycut/smear))
endif

; get data A and B min and max values
minmaxA=minmax(dataA)
minmaxB=minmax(dataB)
minA=minmaxA[0]
maxA=minmaxA[1]
minB=minmaxB[0]
maxB=minmaxB[1]

; check specified plotranges
if (plotrangeA[0] gt maxA or plotrangeA[1] lt minA $
    and max(abs(plotrangeA)) ne 0) then begin
  print,' ===== scatcont wrong plotrangeA, using default'
  plotrangeA=[0,0]
endif
if (plotrangeB[0] gt maxB or plotrangeB[1] lt minB $
    and max(abs(plotrangeB)) ne 0) then begin
  print,' ===== scatcont wrong plotrangeB, using default'
  plotrangeB=[0,0]
endif

; check boxpearson
if (boxrangeA eq [0,0] or boxrangeB eq [0.0]) then boxpearson=0 

; default plotranges 
if (max(abs(plotrangeA)) eq 0) then $
  plotrangeA=[minA-0.1*(maxA-minA),maxA+0.1*(maxA-minA)]

if (max(abs(plotrangeB)) eq 0) then begin
  Boff=0.1+0.06*(quadpearson eq 1)
  plotrangeB=[minB-Boff*(maxB-minB),maxB+0.1*(maxB-minB)]
endif 

; plot frame with erase to start new plot
plotarea=[0.2,0.2,0.95,0.95] 
plot,plotrangeA,plotrangeB,/nodata,$
  position=plotarea,$
  xrange=plotrangeA,xstyle=1,yrange=plotrangeB,ystyle=1,_extra=plotkeywords

; plot frame
plot,plotrangeA,plotrangeB,/nodata,/noerase,$
  position=plotarea,$
  xrange=plotrangeA,xstyle=1,yrange=plotrangeB,ystyle=1,_extra=plotkeywords

;RR from here mostly de Wijn stuff until I say so

; set plot size
realxsize=(!x.window[1]-!x.window[0])*!d.x_size/!d.x_px_cm
realysize=(!y.window[1]-!y.window[0])*!d.y_size/!d.y_px_cm
dx=!x.crange[1]-!x.crange[0]  ;RR length x axis in data units
dy=!y.crange[1]-!y.crange[0]

; nr of bins
nx=nbins
ny=nbins

;RR optional rescale to bin-split data range instead of plotrange 
if (binscale eq 1) then begin
  nx=fix(round(dx/(maxA-minA)*nbins))   ; round gives long integer
  ny=fix(round(dy/(maxB-minB)*nbins))
endif

; binsize in x and y, avoid discretization issues
bx=float(dx)/(nx-1.)
if (size(dataA,/type) ne 4 and size(dataA,/type) ne 5) then begin
  bx=fix(bx)
  nx=fix(dx/bx)+1
endif
by=float(dy)/(ny-1.)
if (size(dataB,/type) ne 4 and size(dataB,/type) ne 5) then begin
  by=fix(by)
  ny=fix(dy/by)+1
endif

; scale data (RR: divide by bin width)
if (!x.crange[0] ne 0) then xd=dataA-!x.crange[0] else xd=dataA
if (bx ne 1) then xd/=bx    ;RR this IDL notation means divide xd by bx
if (!y.crange[0] ne 0) then yd=dataB-!y.crange[0] else yd=dataB
if (by ne 1) then yd/=by    ;RR never too old to learn

; create histogram array
h=long(nx)*fix(yd)+fix(xd)

; set points outside of !x.crange and !y.crange to -1
range=xd ge 0 and xd lt nx and yd ge 0 and yd lt ny
h=(temporary(h)+1)*temporary(range)-1

; compute histogram
h=histogram(h,min=0,max=long(nx)*long(ny)-1l)
h=reform(h,nx,ny,/overwrite)

; quit if h is not 2D
sizeh=size(h)
if (sizeh[1] lt 2 or sizeh[2] lt 2) then begin
  print,' ===== scatcont ABORT: histogram not 2D'
  return
endif

xax=findgen(nx)*bx+!x.crange[0]
yax=findgen(ny)*by+!y.crange[0]

SETNL:
nl=fix(alog10(max(h)/float(outerlevel))/alog10(contourstep))
if (nl lt 2) then begin
  ;; xyouts,0.3,0.5,/norm,charsize=1.5,'too few levels to plot'
  ;; return
  contourstep=contourstep-(contourstep-1.)/2.
  goto, SETNL
endif

densatpos=interpolate(h,temporary(xd),temporary(yd))
wltlevel=where(densatpos lt outerlevel)
if (wltlevel[0] ne -1) then begin

;RR added nocloud option not to plot the outside scatter 
  if (not(nocloud)) then oplot,dataA[wltlevel],dataB[wltlevel],psym=3

;RR plot the contours  
  contour,h,xax,yax,xstyle=13,ystyle=13,/overplot,$
    position=plotarea,$
    levels=contourstep^(findgen(nl+1))*outerlevel
endif

;RR ---- moment curves and distribution curves

if (not(nomoments) or not(nohistograms)) then begin
  rowtot=total(h,1)
  coltot=total(h,2)
endif

if (not(nomoments)) then begin
  common scattercommon,xmomval,xmomind,ymomind,ymomval

  rowxtot=fltarr(ny)
  colytot=fltarr(nx)
  for i=0,ny-1 do rowxtot[i]=total(h[*,i]*xax)
  for i=0,nx-1 do colytot[i]=total(h[i,*]*yax)

;RR changed from gt 0 to avoid weird curve endings
  w=where(rowtot gt outerlevel*contourstep)
  if (w[0] eq -1) then w=where(rowtot gt 0)
  xmomval=rowxtot[w]/rowtot[w]
  xmomind=yax[w]
  xmiavg=avg(xmomind)

;RR changed from gt 0 to avoid weird curve endings
  w=where(coltot gt outerlevel*contourstep)
  if (w[0] eq -1) then w=where(rowtot gt 0)
  ymomval=colytot[w]/coltot[w]
  ymomind=xax[w]
  ymiavg=avg(ymomind)

  ;RR abort when too few moments
  nkill=5
  if (n_elements(xmomval) lt nkill or n_elements(ymomval) lt nkill) then begin
    print,' ##### scatcont abort: too few moment values'
    return
  endif

  xi=transpose([[1.0,0.0],[0.0,1.0]])
  r=[xmiavg,ymiavg]
  powell,r,xi,1e-4,fmin,itmax=1E6,'scfunc'

  xmvint=interpol(xmomval,xmomind,r[0],/spline)
  ymvint=interpol(ymomval,ymomind,r[1],/spline)

  wx=reverse(where(xmomind lt r[0]))
  wy=reverse(where(ymomind lt r[1]))
  if (wx[0] ne -1) then oplot_dl,[xmvint,xmomval[wx]],[r[0],xmomind[wx]],$
    (!x.crange[1]-!x.crange[0])/40.,$
    (!x.crange[1]-!x.crange[0])/30.,/quiet
  if (wy[0] ne -1) then oplot_dl,[r[1],ymomind[wy]],[ymvint,ymomval[wy]],$
    (!y.crange[1]-!y.crange[0])/40.,$
    (!y.crange[1]-!y.crange[0])/30.,/y_units,/quiet

  wx=where(xmomind gt r[0])
  wy=where(ymomind gt r[1])
  if (wx[0] ne -1) then oplot_dl,[xmvint,xmomval[wx]],[r[0],xmomind[wx]],$
    (plotrangeA[1]-plotrangeA[0])/40.,(plotrangeA[1]-plotrangeA[0])/30.,$
    /quiet
  if (wy[0] ne -1) then oplot_dl,[r[1],ymomind[wy]],[ymvint,ymomval[wy]],$
    (plotrangeB[1]-plotrangeB[0])/40.,(plotrangeB[1]-plotrangeB[0])/30.,$
    /y_units,/quiet
endif

;RR oplot histograms along sides (in frame)
;RR dx/5, dy/5 are peak heights in frame size
;RR xax and yax are x and y indices of the bins
if (not(nohistograms)) then begin
  oplot,plotrangeA[1]-rowtot/max(rowtot)*dx/5,yax
  oplot,xax,plotrangeB[1]-coltot/max(coltot)*dy/5
endif

; ----- end of de Wijn stuff, RR additions

; get summit coordinates in data units
hmax=where(h eq max(h))
ijmax=array_indices([nx,ny],hmax,/dimensions)
Amax=xax[ijmax[0]]
Bmax=yax[ijmax[1]]

; put cross at mountain summit (blue plus sign)
if (not(nosummit)) then plots,Amax,Bmax,psym=1,symsize=3,color=cgcolor("blue")

; specify number of pixels at top right
if (not(nonumber)) then begin
  npix=n_elements(dataA)
  npixstr=string(npix,format='(E7.1)')
  xyouts,plotrangeA[1]-0.05*dx,plotrangeB[0]+0.90*dy,$
    /data,alignment=1,npixstr,charsize=1.2
endif

; specify contour step below the above number of pixels
if (not(nostep)) then begin
  stepstr=string(contourstep,format='(I4)')
  xyouts,plotrangeA[1]-0.05*dx,plotrangeB[0]+(0.83-(fontsize/9.)*0.01)*dy,$
    /data,alignment=1,stepstr,charsize=1.2
endif

; add Pearson overall correlation at bottom left
if (fullpearson) then begin
  pearcorr=correlate(dataA,dataB,/double)
  pearcorr=round(100.*pearcorr)/100.
  pearcorrstr=string(pearcorr,format='(f5.2)')
  xyouts,plotrangeA[0]+0.04*dx,plotrangeB[0]+0.04*dy,$
    /data,pearcorrstr,charsize=1.2
endif

;RR print Pearson coefficients for data in four quadrants from summit
;RR need precise summit location since very weighted to the density top
if (quadpearson) then begin
  qpear=fltarr(4)
  for iq=0,3 do begin ; clockwise from lower left
    if (iq eq 0) then indquad=where(dataA lt Amax and dataB lt Bmax) 
    if (iq eq 1) then indquad=where(dataA lt Amax and dataB gt Bmax) 
    if (iq eq 2) then indquad=where(dataA gt Amax and dataB gt Bmax) 
    if (iq eq 3) then indquad=where(dataA gt Amax and dataB lt Bmax) 
    q1A=dataA[indquad]
    q1B=dataB[indquad]
    qpear[iq]=correlate(q1A,q1B,/double)
  endfor
; add matrix in plot
  qpear=round(100.*qpear)/100.
  ;; print,' ===== qpear = ',qpear
  quadloc=[0.83-(fontsize/9.)*0.05,0.11]
  quadposx=fltarr(1)+plotrangeA[0]+quadloc[0]*dx ;RR must be arr, fuck IDL
  quadposy=fltarr(1)+plotrangeB[0]+quadloc[1]*dy
  oplot,quadposx,quadposy,psym=1,symsize=7
  xyouts,quadposx-0.02*dx,quadposy-0.07*dy,alignment=1,/data,$
    string(qpear[0],format='(f5.2)'),charsize=1.2
  xyouts,quadposx-0.02*dx,quadposy+0.025*dy,alignment=1,/data,$
    string(qpear[1],format='(f5.2)'),charsize=1.2
  xyouts,quadposx+0.02*dx,quadposy+0.025*dy,alignment=0,/data,$
    string(qpear[2],format='(f5.2)'),charsize=1.2
  xyouts,quadposx+0.02*dx,quadposy-0.07*dy,alignment=0,/data,$
    string(qpear[3],format='(f5.2)'),charsize=1.2
endif

; add one or more label lines 
if (label[0] ne '') then $
  for ilabel=0,n_elements(label)-1 do $
    xyouts,0.23,0.88-(0.05*ilabel),/norm,label[ilabel],charsize=1.1

; add one or more colored boxes
if (max(abs(boxrangeA)) ne 0 and max(abs(boxrangeB)) ne 0) then begin
  nbox=n_elements(boxrangeA)/2
  if (n_elements(boxrangeB)/2 ne nbox) then begin
    print,' ##### scatcont ABORT: specify'+trimd(nbox)+' boxrangeA'
    return
  endif
  if (boxcolor[0] eq -1) then boxcolor=cgcolor("green")
  if n_elements(boxcolor) ne nbox then begin
    print,' ##### scatcont ABORT: specify colors for'+trimd(nbox)+' boxes'
    return
  endif
  for ibox=0,nbox-1 do begin
    length=boxrangeA[1,ibox]-boxrangeA[0,ibox]
    height=boxrangeB[1,ibox]-boxrangeB[0,ibox]
    tvbox,[length,height],boxrangeA[0,ibox]+length/2.,$
      boxrangeB[0,ibox]+height/2.,/data,color=boxcolor[ibox],thick=boxthick
  endfor
endif

; add Pearson correlation for only data within each box
if (max(boxpearson) eq 1) then begin
  nbox=n_elements(boxrangeA)/2
  nboxpear=n_elements(boxpearson)
  if (nboxpear ne nbox) then begin
    print,' ##### scatcont abort: boxpearson must have nbox dimension'
    return
  endif
  nrpear=-1
  for ibox=0,nbox-1 do begin
    if (boxpearson[ibox] eq 1) then begin
      nrpear=nrpear+1
      indbox=where($
        dataA gt boxrangeA[0,ibox] and dataA lt boxrangeA[1,ibox] and $
        dataB gt boxrangeB[0,ibox] and dataB lt boxrangeB[1,ibox])
      if (indbox[0] eq -1) then begin
        print,' ====== scatcont skips boxpearson: '+$
          'boxrange does not match plotrange'
        goto, SKIPBOXPEARSON
      endif
      boxA=dataA[indbox]
      boxB=dataB[indbox]
      boxpear=correlate(boxA,boxB,/double)
      boxpear=round(100.*boxpear)/100.
      boxpearstr=string(boxpear,format='(f5.2)')
      xyouts,boxrangeA[1,ibox]-0.02*dx,boxrangeB[0]+0.02*dy,$
        /data,alignment=1,boxpearstr,charsize=1.2,color=boxcolor[ibox]
    endif
    SKIPBOXPEARSON:
  endfor 
endif

; undo coyote color coding for not upsetting RGB for what comes after
loadct,0

; close ps file
if (psfile ne '') then closepsplot,psfilename,opengv=0

end

; ================= test per IDLWAVE Hyper-C = also example and template

; define data (DOT cubes on website under manuals)
datapath='/home/rutten/rr/edu/manuals/idl-cubes/idl-cube/' 
dataA=readfits(datapath+'DOT-2005-10-14-gb.fits')
dataB=readfits(datapath+'DOT-2005-10-14-ca.fits') 
xtitle='G-band intensity' 
ytitle='Ca II H intensity'
;; plotrangeA=[-10,190]  ; sticky: rerun IDL or it remembers its last 
;; plotrangeB=[-10,190]
;; label=['asjemenou','hebikooit','godsammekraken']

; inspect data
;; showex,dataA,dataB

; scatcont choices
;; contourstep=2
;; outerlevel=50
;; nocloud=1
;; nomoments=1
;; nohistograms=1
fullpearson=1 
quadpearson=1
;; scaleA=10 ; keep byte values of both images
;; boxrangeA=[[0,50],[100,250]]
;; boxrangeB=[[100,150],[100,240]]
;; boxcolor=[cgcolor('green'),cgcolor('blue')] 
;; boxpearson=[0,1]
boxrangeA=[100,240]
boxrangeB=[90,240]
boxcolor=cgcolor('blue')
boxpearson=1
boxthick=3

; plot choices (NB: sticky within IDL session)
fontsize=10
;; thick=3

; choose ps or screen output 
psfile='/tmp/scatcont.ps'
;; psfile=''
;; window,xsize=500,ysize=500

;; ; block bright points for all time steps
;; blockbox=[130,170,50,150,0,0] 

;; plotrangeA=[-200,600] ; far extended
;; plotrangeB=plotrangeA

scatcont,dataA,dataB,$
  blockbox=blockbox,plotrangeA=plotrangeA,plotrangeB=plotrangeB,$
  boxrangeA=boxrangeA,boxrangeB=boxrangeB,$
  boxcolor=boxcolor,boxthick=boxthick,binscale=binscale,$
  outerlevel=outerlevel,nbins=nbins,contourstep=contourstep,$
  nocloud=nocloud,nomoments=nomoments,nohistograms=nohistograms,$
  nosummit=nosummit,nonumber=nonumber,nostep=nostep,$
  fullpearson=fullpearson,quadpearson=quadpearson,boxpearson=boxpearson,$
  smear=smear,label=label,psfile=psfile,$
  fontsize=fontsize,thick=thick,xtitle=xtitle,ytitle=ytitle,$
  scaleA=scaleA,scaleB=scaleB

; store if screen output
;; scatimage=tvread()

;; ; show if ps output
;; spawn,'killgv'
;; spawn,'gv '+psfile

; interpretation
;   top-right bright-bright correlation = magnetic concentrations
;   mountain reverse correlation = reverse granulation + gravity waves

end
