; file: plotkophasediff.pro = plot k-f phase difference diagram
; init: Sep  5 2010 
; last: Oct 18 2010 
;
;+
; NAME: plotkophasediff
;
; PURPOSE: plot "k-omega" Fourier phase difference diagram 
;   vertical f, horizontal spatial wavenumber k_h
;
; CALLING SEQUENCE:
;   plotkophasediff, cube1, cube2, arcsecpx, cadence, plotfilename, $
;     apod=apod, kmax=kmax, fmax=fmax, $
;     minphasediff=minphasediff, maxphasediff=maxphasediff, $
;     contours=contours, lamb=lamb, fundamental=fundamental
;
; INPUTS:
;   cube1 and cube 2: (x,y,t) data cubes, any type
;   arcsecpx = angular image scale in arcsec/px
;   cadence: [regular] sampling cadence in sec
;   plotfilename: postscript output file name
;     optional keywords: 
;   apod: fractional extent of apodization edges; default 0.1
;   kmax: maximum k_h axis as fraction of Nyquist value, default 0.2
;   fmax: maximum f axis as fraction of Nyquist value, default 0.5
;   minphasediff: lower cutoff for plot, default -180
;   maxphasediff: upper cutoff for plot, default +180
;   contours: set true to plot contours
;   lamb: value > 0 overplots Lamb line omeha = c_s kn at c_s = lamb km/s 
;   fundamental: set true to overplot fundamental mode omega=sqrt(g k_h)
;
; MODIFICATION HISTORY: 
;  Sep 2010: Rob Rutten (RR) assembly of Alfred de Wijn's routines.
;  Okt 2010: RR optional overplots Lamb line and fundamental mode
;-


;-----------------------------------------------------------------------
function avgstd, array, stdev=stdev
    ; get array average + standard deviation
    ;RR Aug 23 2010 found in Mabula Haverkamp's IDL, later also AdW
    ;RR not the same as avg.pro in ssw
    ;RR not the same as avg.pro in Pit Suetterlin DOT software
    ;RR so renamed to avgstd
  avrg = total(array)/n_elements(array)
  stdev = sqrt(total((array-avrg)^2)/n_elements(array))
  return, avrg
end

;----------------------------------------------------------------------------
function linear, x, p   ;RR used in temporal detrending
	ymod = p[0] + x * p[1]
	return, ymod
end

;----------------------------------------------------------------------------
function gradient, x, y, p    ;RR used in spatial detrending
	zmod = p[0] + x * p[1] + y * p[2]
	return, zmod
end

;----------------------------------------------------------------------------
function apod3dcube,cube,apod
    ; apodizes cube in all three coordinates, with detrending 

  ; get cube dimensions
  sizecube=size(cube)
  nx=sizecube[1]
  ny=sizecube[2]
  nt=sizecube[3]
  apocube = fltarr(nx,ny,nt)

  ; define temporal apodization
  apodt = fltarr(nt)+1
  if (apod ne 0) then begin
    apodrimt = nt*apod
    apodt[0] = (sin(!pi/2.*findgen(apodrimt)/apodrimt))^2
    apodt = apodt*shift(rotate(apodt,2),1)   ;RR had ik nooit verzonnen
  endif 

  ; temporal detrending, not per px, only mean-image trend 
  ttrend = fltarr(nt)
  tf = findgen(nt) + 1
  for it=0, nt-1 do begin
    img = cube[*,*,it]
    ttrend[it] = avgstd(img)
  endfor
  fitp = mpfitfun('linear', tf, ttrend, fltarr(nt)+1, [1000.,0.],/quiet)
  fit = fitp[0] + tf * fitp[1]

  ; temporal apodization per (x,y) column
  ;RR do not reinsert trend to keep [0,0] Fourier pixel from dominating
  for it=0, nt-1 do begin
    img = cube[*,*,it]
    apocube[*,*,it] = (img-fit[it])*apodt[it]   ;RR + ttrend[it]
  endfor

  ; define spatial apodization
  apodx = fltarr(nx)+1
  apody = fltarr(ny)+1
  if (apod ne 0) then begin
    apodrimx=apod*nx
    apodrimy=apod*ny
    apodx[0] = (sin(!pi/2.*findgen(apodrimx)/apodrimx))^2
    apody[0] = (sin(!pi/2.*findgen(apodrimy)/apodrimy))^2
    apodx = apodx*shift(rotate(apodx,2),1)
    apody = apody*shift(rotate(apody,2),1)
    apodxy = apodx # apody
 endif

  ; spatial gradient removal + apodizing per image
  xf = fltarr(nx,ny)+1.
  yf = xf
  for it=0, nt-1 do begin
    img = apocube[*,*,it]
    avg = avgstd(img)
    ;RR mpfit2dfun = ssw/gen/idl/fitting/mpfit/mpfit2dfun.pro
    fitp = mpfit2dfun('gradient',xf,yf,img,fltarr(nx,ny)+1,[1000.,0.,0.],$
                      /quiet)
    fit = fitp[0]+xf*fitp[1]+yf*fitp[2]
    apocube[*,*,it] = (img-fit)*apodxy + avg
  endfor

  ; done
  return,apocube
end

;---------------------------------------------------------------------------
function ko_dist, sx, sy, double=double
  ; set up Pythagoras distance array from origin 
  ;RR from Alfred de Wijn email Aug 30 2010 
  dx = rebin(dindgen(sx/2+1)/(sx/2),sx/2+1,sy/2+1)
  dy = rotate(rebin(dindgen(sy/2+1)/(sy/2),sy/2+1,sx/2+1),1)
  dxy = sqrt(dx^2+dy^2)*(min([sx,sy])/2+1.)
  afstanden = dblarr(sx,sy)
  afstanden[0,0] = dxy
  ; get other quadrants
  afstanden[sx/2,0] = rotate(dxy[1:*,*],5)       ;RR 5 = 90 deg
  afstanden[0,sy/2] = rotate(dxy[*,1:*],7)       ;RR 7 = 270 deg
  afstanden[sx/2,sy/2] = rotate(dxy[1:*,1:*],2)  ;RR 2 = 180 deg
  if not keyword_set(double) then afstanden = fix(round(afstanden))
  return, afstanden
end

;---------------------------------------------------------------------------
function averdphi,cube1,cube2
  ; compute 2D (k_h,f) phase difference array by k_x, k_y averaging

  ; get cube dimensions
  sizecube=size(cube1)
  nx=sizecube[1]
  ny=sizecube[2]
  nt=sizecube[3]

  ; forward fft and throw away half of it
  ; perform fft in time direction first
  fftcube1 = (fft(fft((fft(cube1,-1, dimension=3))[*,*,0:nt/2],-1,$
     dimension=1),dimension=2))
  fftcube2 = (fft(fft((fft(cube2,-1, dimension=3))[*,*,0:nt/2],-1,$
     dimension=1),dimension=2))

  ; get crosspower between the cubes
  crosspower = fftcube1*conj(fftcube2)

  ; set up distances 
  fftfmt = size(fftcube1)
  afstanden = ko_dist(nx,ny)   ;RR integer-rounded Pythagoras array
  maxdist = min([nx,ny])/2-1   ;RR largest quarter circle

  ; get averaged phase differences
  avdphi = fltarr(maxdist+1,nt/2+1)
  ; average phasediff over all distances, building the spectrum
  for i=0, maxdist do begin
    waar = where(afstanden eq i)
    for j=0, nt/2 do begin
      w1 = (crosspower[*,*,j])[waar]
      waarden = total(w1)
      avdphi[i,j] = atan(imaginary(waarden),float(waarden))
    endfor
;;  writeu, -1, string(format='(%"\rcomputing... ",i6,"/",i6)', i, maxdist)
  endfor

  ; done
  return, avdphi
end

;---------------------------------------------------------------------------
pro koplotdphi,avdphi,arcsecpx,cadence,kmax,fmax,mindphi,maxdphi,$
      contours,lamb,fundamental,plotfilename
    ; plotting program, this is the hardest part
  sizedphi = size(avdphi)

  ; select extent = fractions of Nyquist values
  plotrange = [fix(sizedphi[1]*kmax),fix(sizedphi[2]*fmax)]
  plotdphi = avdphi[0:plotrange[0]-1,0:plotrange[1]-1]

  ;RR 5x5 resizing, I guess for better tick positioning and contours
  xas = 2.*!pi/(arcsecpx*2)*findgen(plotrange[0])/(sizedphi[1]-1)
  rexas = 2.*!pi/(arcsecpx*2)*findgen(plotrange[0]*5)/(sizedphi[1]*5-1)
  yas = 1./(cadence*2)*findgen(plotrange[1])/(sizedphi[2]-1)*1e3
  reyas = 1./(cadence*2)*findgen(plotrange[1]*5)/(sizedphi[2]*5-1)*1e3
  smoothdphi = convol(rebin(plotdphi,plotrange[0]*5,plotrange[1]*5),$
    fltarr(6,6)+1/(6.*6.),/edge_truncate)/!pi*180
  ;; ;RR old version was:
  ;; smoothdphi = smooth(rebin(plotdphi,plotrange[0]*5,plotrange[1]*5),$
  ;;   12,/edge)/!pi*180    ;RR rather different result,why?
  xrange = [min(xas),max(xas)]
  yrange = [min(yas),max(yas)]
   
  ; plot startup: what windows?
  windowsdevice='X'
  if (!d.name eq 'WIN') then windowsdevice='WIN'

  ; insert of Alfred's startplot.pro choices
  set_plot, 'ps'
  !p.font=1
  !p.thick=3
  !x.thick=3
  !y.thick=3
  !z.thick=3
  aspect=3./2
  device,filename=plotfilename,xsize=7,ysize=7,$
    set_font='Times', /tt_font,font_size=10,bits_per_pixel=8

  ; plot phase diff image; not full grayscale to leave black for contours
  smoothdphi=smoothdphi > mindphi < maxdphi
  tv, bytscl(smoothdphi,top=255-64)+64, $
    0.15, 0.15, /normal, xsize=0.7, ysize=0.48

  ; define plot frame for overplots
  plot, xrange, yrange, /nodata, xstyle=13, ystyle=13,$
    position=[0.15,0.15,0.85,0.63], /noerase

  ; overplot contours if requested
  if (contours ne 0) then begin
    contour, smoothdphi,rexas,reyas, $
      xstyle=9, ystyle=9, /overplot, $
      levels=[-3./2,-5./4,-1,-3./4,-1./2,-1./4,0,1./4,1./2]*180, $
      closed=0, c_labels=[0], c_thick=[2,1,2,1,2,1,2,1,2]
 endif

  ; overplot Lamb line (horizontal propagation at Lamb value = sound speed)
  ; 725 is km/arcsec on the sun at mean distance from Earth 
  ; 1/2pi from omega to f; 1E3 from Hz to mHz
  if (lamb ne 0) then oplot, rexas, 1./(2.*!pi)*1E3*lamb/725.*rexas, color=255

  ; overplot fundamental mode 
  ; g = 2.74E4 cm/s^2 = 2.74E-1 km/s^2 = 2.741E-1/725. arcsec/s^2
  if (fundamental ne 0) then $
     oplot, rexas, 1./(2.*!pi)*1E3*sqrt(2.741E-1/725.*rexas),$
       color=255, linestyle= 2

  ; plot x,y axes
  plot, xrange, yrange, /nodata, xstyle=9, ystyle=9,$
    position=[0.15,0.15,0.85,0.63], $
    yticklen=-0.015/0.7, xticklen=-0.015/0.53, /noerase, $
    xminor=1, xtitle='horizontal wavenumber [arcsec!U-1!N]', $
    yminor=1, ytitle='frequency [mHz]'

  ; plot wavelength axis along top
  if (xrange[1] le 10) then begin ;RR I wonder how to automate this  
    wavtickspos = [10, 5, 3, 2, 1.5, 1]
    wavticksn = ['10','5','3','2','1.5','1']
  endif else if (xrange[1] le 20) then begin
    wavtickspos = [10, 5, 3, 2, 1.5, 1, 0.5]
    wavticksn = ['10','5','3','2','1.5','1','0.5']
  endif else if (xrange[1] le 50) then begin
    wavtickspos = [5.0, 2.0, 1.0, 0.5, 0.2]
    wavticksn = ['5','2','1','0.5','0.2']
  endif else begin
    wavtickspos = [5.0, 2.0, 1.0, 0.5, 0.2, 0.1, 0.05]
    wavticksn = ['5','2','1','0.5','0.2','0.1','0.05']
  endelse
  wavticks=n_elements(wavtickspos)-1
  wavticksv = 2.*!pi/wavtickspos   ;RR wavelength from circle frequency
  axis, /xaxis, xticks=wavticks, xtickv=wavticksv, xtickname=wavticksn, $
    ticklen=-0.015/0.53, xminor=1, xtitle='wavelength [arcsec]'

  ; plot period axis along righthand side 
  if (yrange[1] le 10) then begin ;RR I wonder how to automate this 
    pertickspos = [20, 10, 5, 3, 2, 1]
    perticksn = ['20','10','5','3','2','1']
  endif else if (yrange[1] le 20) then begin
    pertickspos = [10, 5, 3, 2, 1.5, 1, 0.5]
    perticksn = ['10', '5','3','2','1.5','1','0.5']
  endif else if (yrange[1] le 50) then begin
    pertickspos = [10, 5, 2, 1, 0.5, 0.2, 0.1]
    perticksn = ['10','5','2','1','0.5','0.2','0.1']
  endif else begin
    pertickspos = [2, 1, 0.5, 0.2, 0.1]
    perticksn = ['2','1','0.5','0.2','0.1']
  endelse
  perticks=n_elements(pertickspos)-1
  perticksv = 1e3/pertickspos/60.  ;RR period, from mHz to min
  axis, /yaxis, yticks=perticks, ytickv=perticksv, ytickname=perticksn, $
    ticklen=-0.015/0.7, yminor=1, ytitle='period [min]'

  ; add grayscale bar on top
  tv,  findgen(255-64)+64, 0.15, 0.91, /normal, xsize=0.7, ysize=0.02
  plot, [mindphi,maxdphi], [0,1], /noerase, /nodata,$
    position=[0.15,0.91,0.85,0.93],$
    xrange=[mindphi,maxdphi], xticks=6,$
    xstyle=1, ystyle=1, yticklen=0.0, xticklen=-0.015/0.02, $
    xminor=1, xtitle='phase difference [degrees]',$
    yminor=1,yticks=1,ytickname=[' ',' ']

  ; done 
  device, /close
  set_plot, windowsdevice

  ; convert the IDL ad so that gv shows the filename as window label
  if (windowsdevice eq 'X') then begin
    spawn,'cat '+plotfilename+$
      '| sed "s|Graphics produced by IDL|'+plotfilename+$
      '|" >  idltemp.ps; mv idltemp.ps '+plotfilename
  endif
end

; --------------------------- main part ------------------------------

pro plotkophasediff,cube1,cube2,arcsecpx,cadence,plotfilename,$
  apod=apod,kmax=kmax,fmax=fmax,$
  minphasediff=minphasediff,maxphasediff=maxphasediff,$
  contours=contours,lamb=lamb,fundamental=fundamental
    ; wrapper calling the above subroutines

if (n_elements(apod) ne 0) then apod=apod else apod=0.1
if (n_elements(kmax) ne 0) then kmax=kmax else kmax=1.0
if (n_elements(fmax) ne 0) then fmax=fmax else fmax=1.0
if (n_elements(minphasediff) ne 0) $
   then mindphi=minphasediff else mindphi=-180.
if (n_elements(maxphasediff) ne 0) $
   then maxdphi=maxphasediff else maxdphi=+180.
if not keyword_set(contours) then contours=0 else contours=1
if (n_elements(lamb) ne 0) then lamb=lamb else lamb=7.
if not keyword_set(fundamental) then fundamental=0 else fundamental=1

if (kmax gt 1) then kmax=1
if (fmax gt 1) then fmax=1

if n_params() lt 5 then begin
    print, ' plotkophasediff,cube1,cube2,arcsecpx,cadence,plotfilename,'
    print, '   apod=apod (default 0.1),'
    print, '   kmax=kmax (default 1.0),'
    print, '   fmax=fmax (default 1.0),'
    print, '   minphasediff=minphasediff (default -180),'
    print, '   maxphasediff=maxphasediff (default +180),'
    print, '   contours=contours (default 0)'
    print, '   lamb=lamb (sound speed km/s, default 7)'
    print, '   fundamental=fundamental (default 0)'
  return
endif

; apodize the cubes
apocube1=apod3dcube(cube1,apod)
apocube2=apod3dcube(cube2,apod)

; compute radially-averaged phasediff
avdphi=averdphi(apocube1,apocube2)

; plot the diagram
koplotdphi,avdphi,arcsecpx,cadence,kmax,fmax,mindphi,maxdphi,$
  contours,lamb,fundamental,plotfilename

; done
print,'  wrote file ', plotfilename

end
