; file: cube_rotate.pro
; init: Jun 12 2014  Rob Rutten  Deil
; last: Jun 19 2014  Rob Rutten  Deil

function cube_rotate,incube,angle,fullfield=fullfield,missing=missing

;+
 ; NAME: 
 ;   cube_rotate.pro
 ;   
 ; PURPOSE: 
 ;   function to rotate a cube, eg image cube = movie
 ;   
 ; DESCRIPTION: 
 ;   why didn't I find such a simple program in SSW?
 ;   
 ; CALL: 
 ;   rotcube=cube_rotate(incube,angle,fullfield=fullfield,missing=missing)
 ;   
 ; INPUTS: 
 ;   incube = 3D data cube of any type, eg movie [nx,ny,nt] or CRISP cube
 ;   angle = angle to rotate over, in degrees clockwise
 ;           either a constant or a [nt] vector specifying angle per frame
 ;   
 ; OPTIONAL INPUTS:
 ;   fullfield =  1/0: extend array size to accommodate rotated corners  
 ;   missing = data value for rotated-in blank pixels
 ;     default: missing=avg(incube)
 ;     NB: missing=0 gives the default; use eg missing=0.001 to get black
 ;   
 ; OUTPUTS: 
 ;   rotated cube, same type as incube, same size if \fullfield
 ;   is not called but symmetrically extended in x and y for /fullfield
 ;   
 ; RESTRICTIONS: 
 ;   if the cubes exceed memory use rotatefitscube.pro instead
 ;
 ; HISTORY: 
 ;   Jun 13 2014 RR: start
 ;   Jun 19 2014 RR: added fullfield option
;-

; answer a no-parameter query
if (n_params() lt 2) then begin
  print,' -- rotcube=rotatecube(incube,angle,fullfield,missing)'
  print,' -- angle: degrees clockwise, either constant or [nt] vector'
  print,' -- fullfield: set to expand image size to accommodate corners'
  print,' -- missing = data value rotated-in blank px, default mean(incube)'
  print,' -- (use missing = 0.001 or such to get black, not missing=0)'
  return,-1
endif

; start 
sizecube=size(incube)
nx=sizecube[1]
ny=sizecube[2]
nt=sizecube[3]
datatype=sizecube[4]

; check angle specification
if (n_elements(angle) ne 1 and n_elements(angle) ne nt) then begin
  print,' ===== cube_rotate ERROR: angle not a constant nor [nt] vector'
  return,-1
endif

; default for missing
if (n_elements(missing) eq 0) then missing=avg(incube)

; extend image plane for fullfield option (no rotated corner cutoffs) 
if (n_elements(fullfield) eq 0) then outcube=incube else begin
  dx=intarr(nt)
  dy=intarr(nt)
  semidiag=sqrt(float(nx-1)^2+float(ny-1)^2)/2.
  angdiag=acos((nx-1)/2./semidiag)*!radeg
  for it=0,nt-1 do begin
    dx[it]=fix(cos((angle[it]-angdiag)*!dtor)*semidiag-(nx-1)/2.+1) 
    dy[it]=fix(sin((angle[it]+angdiag)*!dtor)*semidiag-(ny-1)/2.+1) 
    if (n_elements(angle) eq 1) then break
  endfor
  dxmax=max(dx)
  dymax=max(dy)
  if (datatype eq 1) then outcube=bytarr(nx+2*dxmax,ny+2*dymax,nt)$
    +byt(missing)
  if (datatype eq 2) then outcube=intarr(nx+2*dxmax,ny+2*dymax,nt)$
    +fix(missing)
  if (datatype eq 3) then outcube=lonarr(nx+2*dxmax,ny+2*dymax,nt)$
    +long(missing)
  if (datatype eq 4) then outcube=fltarr(nx+2*dxmax,ny+2*dymax,nt)$
    +float(missing)
  if (datatype eq 5) then outcube=dblarr(nx+2*dxmax,ny+2*dymax,nt)$
    +double(missing)
  outcube[dxmax:nx-1+dxmax,dymax:ny-1+dymax,*]=incube[*,*,*]
endelse

; rotate frame by frame
ang=angle[0]
for it=0,nt-1 do begin
  if (n_elements(angle) ne 1) then ang=angle[it] 
  outcube[*,*,it]=rot(outcube[*,*,it],ang,cubic=-0.5,missing=missing)
endfor

return,outcube
end


; ================== main for IDLWAVE test ======================

incube=readfits('/home/rutten/data/SDO/2014-01-10-test/cubes/a304.fits')
sizecube=size(incube)
angle=indgen(sizecube[3])+10   ; time dependent rotation 
rotcube=cube_rotate(incube,angle,/fullfield,missing=0.001)
showex,rotcube,mag=2

end
