;-------------------------------------------------------------
;+
; NAME:
;      SCALE
; PURPOSE:
;      This function uses the rebin function to more easily rebin an array 
;      inputing only the integer scale factors and not the rescaled 
;      array sizes. (IDL only allows up to eight dimensions.)
; CATEGORY:
;      IMAGE PROCESSING
; CALLING SEQUENCE:
;      narr = scale(arr,s0,s1,s2,...)
; INPUTS:
;      arr = The array that you wish to rebin.
;        type: array, any type
;      s0 = The integer scale factor in the x dimension.
;        type: scalar,integer ;RR or factor, e.g. 0.5, 0.2
;      s1 = The integer scale factor in the y dimension.
;        type: scalar,integer
;      s2 = The integer scale factor in the z dimension.
;        type: scalar,integer
;      ... = However many dimensions there are (up to eight).
; KEYWORD PARAMETERS:
;      /SAMPLE = Same as in Rebin :
;        Normally REBIN uses bilinear interpolation when magnifying 
;        and neighborhood averaging when minifying. However, if 
;        SAMPLE is present and non-zero, nearest neighbor sampling
;        is used in both cases. Interpolation gives higher quality
;        results but requires more time. 
; OUTPUTS:
;      narr = The array that has been scaled to the desired size.
;        type: array,floating point,fltarr(nx,ny)
; COMMON BLOCKS:
; NOTES:
; MODIFICATION HISTORY:
;      H. Cohl,  21 Jan, 1991 ---  Works for any number of dimensions.
;      H. Cohl,  5 Nov, 1990  ---  Works for an image.
;-
;-------------------------------------------------------------

function scale,arr,s0,s1,s2,s3,s4,s5,s6,s7,sample=sample,help=help

  ;Display idl header if help is required.
  if keyword_set(help) or n_params() lt 2 then begin
    get_idlhdr,'scale.pro'
    narr=-1
    goto,finishup
  endif

  ;Determine size and number of parameters passed to function.
  ssz=size(arr) & np=n_params()

  ;Test to see if sample is called.
  if n_elements(sample) eq 0 then sample=0

  ;Test to see if there is an inconsistency between number of scalings
  ;desired and number of dimensions in array given.
  if (ssz(0) ne np-1) then begin
    print,'Number of scalings and Number of dimensions of array are not equal'
    goto,finishup
  endif

  ;Set up neccesary arrays and constants.
  s=fltarr(np-1) & scl=s & str=''

  ;Loop through number of scalings.
  for i=0,np-2 do begin
    si=strcompress(string(i),/rem)
    q1=execute('s(i)=s'+si)
    scl(i)=ssz(i+1)*s(i)
    str=str+',scl('+si+')'
  endfor

  ;Convert array to floating point.
  farr=float(arr)

  ;Rebin array.
  q2=execute('narr=rebin(farr'+str+',sample=sample)')

  finishup:

return,narr

end
