; file rigidalign.pro
; init: Feb  6 2017  from Pradeep Chitta (MPS Goettingen)
; last: Feb  6 2017  Rob Rutten  Deil

;+

PRO rigidalign1, data,				                 $
  DX=dx,    DY=dy,     X0=x0,    Y0=y0,         $
  NT=nt,    I0=i0,     I1=i1,    shft=shft,     $
  DISPVECS=dispvecs,   max_corr=max_corr,       $
  QUIET=quiet,         VERBOSE=verbose

; NAME:
;	RIGIDALIGN1
; PURPOSE:
;	First calculates coarse 2-D shifts in a series of images
;       by calculating the correlation between the 
;       images with 
;	CORR = (1/N-1)*TOTAL(((F(X,Y)-MEAN(F(X,Y)))/SIGMA(F(X,Y)))*$
;	       ((G(X,Y)-MEAN(G(X,Y)))/SIGMA(G(X,Y))))
;
;       Then a set of finer shifts (at a level of 0.01 pixel) are 
;       calculated by first locating the cross-correlation maximum 
;       and then fitting a polynomial to this peak with IDL's SFIT.
;	The maximum of each correlation matrix is then retrieved at a 
;       finer resolution.
;
; CALLING SEQUENCE:
;	RIGIDALIGN1, data
;
; INPUT:
;     DATA: [nx,ny,nfiles] image time series data cube.
;
; OUTPUT:
;      DISPVECS: File containing the offsets
;      MAX_CORR: Array of the maximum of the calculated correlation (optional)
;
; OPTIONAL KEYWORD INPUT: 
; 
;     DX: the x-dimension of the box in which image-to-image
;         correlations are calculated.  Default = half the size in X
;         (=X/2).
;
;     DY: the y-dimension of the box in which image-to-image
;         correlations are calculated.  Default = half the size in Y
;         (=Y/2).
;
;     X0: the x-coordinate of the lower-left corner of the correlation box.
;         Default = DX/2.
;         
;     Y0: the y-coordinate of the lower-left corner of the correlation box.
;         Default = DY/2.
;         
;     NT: the number of images correlated at one time, i.e. the number
;         of images assembled into a subgroup and aligned with respect
;         to the first image in the group.  Default = 4. Set this
;         parameter to be less than the number of images over which
;         the correlated structures in the images change
;         significantly.
;
;     I0: the series number of the image to be used as the anchor, or
;         reference, image.  Default = 0.
;
;     I1: the series number of the last image in the set to be aligned
;         to the reference image.  Default = nfiles-1.
;
;     SHFT: allowed shifts in X and Y to locate the best offsets
;         Default = DX/4.
;
;     QUIET: if set, all output is suppressed.
;
;     VERBOSE: if set, extra diagnostic output may be printed to STDOUT.
;     
; NOTES:	
;     1. This procedure closely follows FG_RIGIDALIGN.PRO available in SSW
;     in a way the variables are named and the layout is also hacked from
;     that procedure.
;     2. Uses IDL's TOTAL, MEAN, STDDEV, SFIT procedures.
; 
;
; HISTORY: Version 1.0 written by L. P. Chitta (chitta@mps.mpg.de) for
;     the work 2012ApJ...752...48C Version 1.1 added keyword /NAN
;   Feb  6 2017 RR: gotten from Pradeep Chitta   	
;-

t0 = SYSTIME(1)
crlf = STRING(10B)+STRING(13B)

; Name, version, and timing information
prognam = 'RIGIDALIGN1.PRO'
progver = 'V1.1'

loud = 1 - KEYWORD_SET(quiet)
verbose = KEYWORD_SET(verbose)
if (verbose eq 1) then loud = 1
if (loud) then PRINT, 'Running ', prognam

;Determine image size and number of images
sz = SIZE(data)
xs = sz[1]
ys = sz[2]
nfiles = sz[3]

;Set defaults values
if ~ KEYWORD_SET(i0) then i0=0
if ~ KEYWORD_SET(i1) then i1=nfiles-1
if ~ KEYWORD_SET(dx) then dx=xs/2
if ~ KEYWORD_SET(dy) then dy=ys/2
if ~ KEYWORD_SET(x0) then x0=dx/2
if ~ KEYWORD_SET(y0) then y0=dy/2
if ~ KEYWORD_SET(shft) then shft=dx/4 
;applies the same shift in both the directions
if ~ KEYWORD_SET(nt) then nt=4


sh=shft ; a new variable to shorten shft
sh2=sh/2

ng = nfiles/nt     ;number of groups of files 
nr = nfiles mod nt ;remainder at end of data series

docalcs=1 ; always 1 in the current version

;Calculate displacements
if (docalcs) then begin
  t0c = SYSTIME(1)
  if (loud) then MESSAGE,/info,'Calculating rigid alignments...'
  dispvecs = FLTARR(2,nfiles)
  max_corr=FLTARR(nfiles)	
  xlo = (x0) > 0
  xhi = (x0 + dx - 1) < (xs-1)
  ylo = (y0) > 0
  yhi = (y0 + dy - 1) < (ys-1)
  dx = xhi - xlo + 1
  dy = yhi - ylo + 1
  ntfac = (nt-1) > 1           ;nt-1 does the overlap of series
  norm = n_elements(data(xlo:xhi,ylo:yhi,0))-1 
     ; Factor to normalize the total correlation calculated at each step
  imx = 0
  for jj=i0,i1,ntfac do begin   
    if MAX(imx) eq i1 then break
    tj0 = SYSTIME(1)     
    imx = (INDGEN(nt) + jj) < i1 ;file numbers for the group data cube
    imx = imx[UNIQ(imx)]
    nf = N_ELEMENTS(imx)
    dtmp = FLTARR(xs,ys,nf)

    for i=0,nf-1 do begin
      dtmp(*,*,i) = data(*,*,imx(i))
    endfor
    ref = (dtmp(xlo:xhi,ylo:yhi,0)-mean(dtmp(xlo:xhi,ylo:yhi,0),/nan))$
      /stddev(dtmp(xlo:xhi,ylo:yhi,0),/nan)
    c_temp=fltarr(sh,sh,nf)

    for i=0,nf-1 do begin
      for j=0,sh-1 do begin 
        for k=0,sh-1 do c_temp(j,k,i) = $
          total((ref)*((dtmp(xlo+j-sh2:xhi+j-sh2,ylo+k-sh2:yhi+k-sh2,i)$
          -mean(dtmp(xlo+j-sh2:xhi+j-sh2,ylo+k-sh2:yhi+k-sh2,i),/nan))/$
          stddev(dtmp(xlo+j-sh2:xhi+j-sh2,ylo+k-sh2:yhi+k-sh2,i),/nan)),$
          /nan)/norm
      endfor
    endfor
    
; Loop for locating the maximum correlation, then using SFIT to locate
; the fine position of the correlation maximum within 5 x 5 grid
    x_g=(findgen(401))*0.01
    y_g=(findgen(401))*0.01
    tmp=fltarr(2,nf) ; Contains the calculated offsets 
    corr_grid_max=fltarr(nf)
    corr_max=fltarr(nf)
    for hh = 0, nf-1 do begin
      corr_max(hh)=max(c_temp(*,*,hh),index1)
      x_index=index1 mod sh
      y_index=index1  /  sh
      corr_fit=SFIT(reform(c_temp(x_index-2:x_index+2,y_index-2:y_index+2,hh)),4,kx=kx1)
      corr_grid=fltarr(401,401)
      for iii=0,400 do begin 
        for jjj=0,400 do begin 
          for k=0,4 do begin 
            for l=0,4 do corr_grid(iii,jjj)=corr_grid(iii,jjj)$
               +(kx1(l,k)*(x_g(iii)^k)*(y_g(jjj)^l))
          endfor
        endfor
      endfor	
; Locating the maximum from the above fine grid (401 x 401)
      corr_grid_max(hh) = max(corr_grid,index2)
      x_col = index2 mod 401
      y_row=  index2  /  401
      
      tmp(0,hh)=x_g(x_col) - 2.000000 + float(x_index) - float(sh2)  
      tmp(1,hh)=y_g(y_row) - 2.000000 + float(y_index) - float(sh2)
      
    endfor					
    

; now add the displacement of the first image in the set
    if jj gt i0 then for k=0,nf-1 do begin

      tmp[0,k] += dispvecs[0,imx[0]]
      tmp[1,k] += dispvecs[1,imx[0]]
      
    endfor
    
    dispvecs[0,imx] = tmp[0,*]
    dispvecs[1,imx] = tmp[1,*]
    max_corr[imx]   =	corr_grid_max(*)     
    dt = SYSTIME(1) - tj0
    if (loud) then begin
      MESSAGE,/info,'Number of images in displacement set:'+STRTRIM(nf,2)
      MESSAGE,/info,'Time for current displacement set: '+STRTRIM(dt,2)+' seconds.'
    endif

  endfor  ;of jj loop
  datatmp=0
  dt = (SYSTIME(1)-t0c)
  if (loud) then MESSAGE,/info,crlf+'     Total time for rigid alignment calculation: '+STRTRIM(dt,2)+' seconds.'


endif  ;of docalcs block
for j=0,nfiles-1 do print,dispvecs[0,j],dispvecs[1,j]

RETURN
END

;  =================== test per Hyper-C ==================================

;; datadir='/home/rutten/data/SDO/2014-06-14-benchmark/trial19/center/cubes/'
;; cube1=readfits(datadir+'aia131.fits')  ; tough although look alike
;; cube2=readfits(datadir+'aia211.fits')  ; best value approx (0.4,-1.05)
;; im1=cube1[*,*,0]
;; im2=cube2[*,*,0]

datadir='/home/rutten/data/SDO/2014-06-14-benchmark/trial19/center/'
im1=readfits(datadir+'im_aia_131.fits')
im2=readfits(datadir+'im_aia_211.fits')
data=[[[im1]],[[im2]]]
rigidalign1,data,dispvecs=shift,$
  DX=1100,DY=1100,x0=30,y0=30,shft=10

; without params
; 2nd iteration: 0.07  1.01  ;RR opposite sign?  took VERY long
;RR Pradeep got (x,y)=(-0.56,-0.66) looks good
;RR my tile=10 gives (-0.11,-0.60) also looks good

; with params fast response:  0.700000     0.800000
;RR opposite sign of needed shift?

; check insert
  ;; shiftim1=shift_img(im1,-shift)
  ;; imblink,sqrt(histo_opt_rr(shiftim1,1.E-4)),sqrt(histo_opt_rr(im2,1.E-4))
  ;; imblink,im1,shiftim1

end
